/*
	PKGNKEYS.h
*/

/*
	Public Key signatures - GeNerate secret and public KEYS
*/


/*
	SLOP_BITS is how many "carry bits" to allow for intermediate
	calculation results to exceed the size of the modulus.
	It is used by modexp to give some overflow elbow room
	for modmult to use to perform modulo operations with the
	modulus. The number of slop bits required is determined
	by the modmult algorithm.  The Russian peasant modmult
	algorithm only requires 1 slop bit, for example.  Note
	that if we use an external assembly modmult routine,
	SLOP_BITS may be meaningless or may be defined in a
	non-constant manner.
*/

/* Define C names for Russian peasant modmult primitives. */
#define SLOP_BITS 1

/* ----- start of former genprime.c ----- */

#if DebugCheck
#define SHOWPROGRESS
#endif

/* Define some error status returns for keygen... */
#define NOPRIMEFOUND -14        /* slowtest probably failed */
#define NOSUSPECTS -13                /* fastsieve probably failed */

/* #define STRONGPRIMES */
	/* if defined, generate "strong" primes for key */
/*
	"Strong" primes are no longer advantageous, due to the
	new elliptical curve method of factoring.  Randomly
	selected primes are as good as any.  See "Factoring", by
	Duncan A. Buell, Journal of Supercomputing 1 (1987),
	pages 191-216. This justifies disabling the lengthy
	search for strong primes.

	The advice about strong primes in the early RSA
	literature applies to 256-bit moduli where the attacks
	were the Pollard rho and P-1 factoring algorithms.
	Later developments in factoring have entirely supplanted
	these methods.  The later algorithms are always faster
	(so we need bigger primes), and don't care about
	STRONGPRIMES.

	The early literature was saying that you can get away
	with small moduli if you choose the primes carefully.
	The later developments say you can't get away with small
	moduli, period.  And it doesn't matter how you choose
	the primes.

	It's just taking a heck of a long time for the advice on
	"strong primes" to disappear from the books.  Authors
	keep going back to the original documents and repeating
	what they read there, even though it's out of date.
*/

#define BLUM
/*
	If BLUM is defined, this looks for prines congruent to 3
	modulo 4. The product of two of these is a Blum
	integer.  You can uniquely define a square root Cmodulo
	a Blum integer, which leads to some extra possibilities
	for encryption algorithms.  This shrinks the key space
	by 2 bits, which is not considered significant.
*/


/*
	primetable is a table of 16-bit prime numbers used for sieving
	and for other aspects of public-key cryptographic key generation
*/

LOCALVAR const ui4b primetable[] =
{
	2, 3, 5, 7, 11, 13, 17, 19,
	23, 29, 31, 37, 41, 43, 47, 53,
	59, 61, 67, 71, 73, 79, 83, 89,
	97, 101, 103, 107, 109, 113, 127, 131,
	137, 139, 149, 151, 157, 163, 167, 173,
	179, 181, 191, 193, 197, 199, 211, 223,
	227, 229, 233, 239, 241, 251, 257, 263,
	269, 271, 277, 281, 283, 293, 307, 311,
	313, 317, 331, 337, 347, 349, 353, 359,
	367, 373, 379, 383, 389, 397, 401, 409,
	419, 421, 431, 433, 439, 443, 449, 457,
	461, 463, 467, 479, 487, 491, 499, 503,
	509, 521, 523, 541, 547, 557, 563, 569,
	571, 577, 587, 593, 599, 601, 607, 613,
	617, 619, 631, 641, 643, 647, 653, 659,
	661, 673, 677, 683, 691, 701, 709, 719,
	727, 733, 739, 743, 751, 757, 761, 769,
	773, 787, 797, 809, 811, 821, 823, 827,
	829, 839, 853, 857, 859, 863, 877, 881,
	883, 887, 907, 911, 919, 929, 937, 941,
	947, 953, 967, 971, 977, 983, 991, 997,
	1009, 1013, 1019, 1021, 1031, 1033, 1039, 1049,
	1051, 1061, 1063, 1069, 1087, 1091, 1093, 1097,
	1103, 1109, 1117, 1123, 1129, 1151, 1153, 1163,
	1171, 1181, 1187, 1193, 1201, 1213, 1217, 1223,
	1229, 1231, 1237, 1249, 1259, 1277, 1279, 1283,
	1289, 1291, 1297, 1301, 1303, 1307, 1319, 1321,
	1327, 1361, 1367, 1373, 1381, 1399, 1409, 1423,
	1427, 1429, 1433, 1439, 1447, 1451, 1453, 1459,
	1471, 1481, 1483, 1487, 1489, 1493, 1499, 1511,
	1523, 1531, 1543, 1549, 1553, 1559, 1567, 1571,
	1579, 1583, 1597, 1601, 1607, 1609, 1613, 1619,
	1621, 1627, 1637, 1657, 1663, 1667, 1669, 1693,
	1697, 1699, 1709, 1721, 1723, 1733, 1741, 1747,
	1753, 1759, 1777, 1783, 1787, 1789, 1801, 1811,
	1823, 1831, 1847, 1861, 1867, 1871, 1873, 1877,
	1879, 1889, 1901, 1907, 1913, 1931, 1933, 1949,
	1951, 1973, 1979, 1987, 1993, 1997, 1999, 2003,
	2011, 2017, 2027, 2029, 2039, 2053, 2063, 2069,
	2081, 2083, 2087, 2089, 2099, 2111, 2113, 2129,
	2131, 2137, 2141, 2143, 2153, 2161, 2179, 2203,
	2207, 2213, 2221, 2237, 2239, 2243, 2251, 2267,
	2269, 2273, 2281, 2287, 2293, 2297, 2309, 2311,
	2333, 2339, 2341, 2347, 2351, 2357, 2371, 2377,
	2381, 2383, 2389, 2393, 2399, 2411, 2417, 2423,
	2437, 2441, 2447, 2459, 2467, 2473, 2477, 2503,
	2521, 2531, 2539, 2543, 2549, 2551, 2557, 2579,
	2591, 2593, 2609, 2617, 2621, 2633, 2647, 2657,
	2659, 2663, 2671, 2677, 2683, 2687, 2689, 2693,
	2699, 2707, 2711, 2713, 2719, 2729, 2731, 2741,
	2749, 2753, 2767, 2777, 2789, 2791, 2797, 2801,
	2803, 2819, 2833, 2837, 2843, 2851, 2857, 2861,
	2879, 2887, 2897, 2903, 2909, 2917, 2927, 2939,
	2953, 2957, 2963, 2969, 2971, 2999, 3001, 3011,
	3019, 3023, 3037, 3041, 3049, 3061, 3067, 3079,
	3083, 3089, 3109, 3119, 3121, 3137, 3163, 3167,
	3169, 3181, 3187, 3191, 3203, 3209, 3217, 3221,
	3229, 3251, 3253, 3257, 3259, 3271, 3299, 3301,
	3307, 3313, 3319, 3323, 3329, 3331, 3343, 3347,
	3359, 3361, 3371, 3373, 3389, 3391, 3407, 3413,
	3433, 3449, 3457, 3461, 3463, 3467, 3469, 3491,
	3499, 3511, 3517, 3527, 3529, 3533, 3539, 3541,
	3547, 3557, 3559, 3571, 3581, 3583, 3593, 3607,
	3613, 3617, 3623, 3631, 3637, 3643, 3659, 3671,
	3673, 3677, 3691, 3697, 3701, 3709, 3719, 3727,
	3733, 3739, 3761, 3767, 3769, 3779, 3793, 3797,
	3803, 3821, 3823, 3833, 3847, 3851, 3853, 3863,
	3877, 3881, 3889, 3907, 3911, 3917, 3919, 3923,
	3929, 3931, 3943, 3947, 3967, 3989, 4001, 4003,
	4007, 4013, 4019, 4021, 4027, 4049, 4051, 4057,
	4073, 4079, 4091, 4093, 4099, 4111, 4127, 4129,
	4133, 4139, 4153, 4157, 4159, 4177, 4201, 4211,
	4217, 4219, 4229, 4231, 4241, 4243, 4253, 4259,
	4261, 4271, 4273, 4283, 4289, 4297, 4327, 4337,
	4339, 4349, 4357, 4363, 4373, 4391, 4397, 4409,
	4421, 4423, 4441, 4447, 4451, 4457, 4463, 4481,
	4483, 4493, 4507, 4513, 4517, 4519, 4523, 4547,
	4549, 4561, 4567, 4583, 4591, 4597, 4603, 4621,
	4637, 4639, 4643, 4649, 4651, 4657, 4663, 4673,
	4679, 4691, 4703, 4721, 4723, 4729, 4733, 4751,
	4759, 4783, 4787, 4789, 4793, 4799, 4801, 4813,
	4817, 4831, 4861, 4871, 4877, 4889, 4903, 4909,
	4919, 4931, 4933, 4937, 4943, 4951, 4957, 4967,
	4969, 4973, 4987, 4993, 4999, 5003, 5009, 5011,
	5021, 5023, 5039, 5051, 5059, 5077, 5081, 5087,
	5099, 5101, 5107, 5113, 5119, 5147, 5153, 5167,
	5171, 5179, 5189, 5197, 5209, 5227, 5231, 5233,
	5237, 5261, 5273, 5279, 5281, 5297, 5303, 5309,
	5323, 5333, 5347, 5351, 5381, 5387, 5393, 5399,
	5407, 5413, 5417, 5419, 5431, 5437, 5441, 5443,
	5449, 5471, 5477, 5479, 5483, 5501, 5503, 5507,
	5519, 5521, 5527, 5531, 5557, 5563, 5569, 5573,
	5581, 5591, 5623, 5639, 5641, 5647, 5651, 5653,
	5657, 5659, 5669, 5683, 5689, 5693, 5701, 5711,
	5717, 5737, 5741, 5743, 5749, 5779, 5783, 5791,
	5801, 5807, 5813, 5821, 5827, 5839, 5843, 5849,
	5851, 5857, 5861, 5867, 5869, 5879, 5881, 5897,
	5903, 5923, 5927, 5939, 5953, 5981, 5987, 6007,
	6011, 6029, 6037, 6043, 6047, 6053, 6067, 6073,
	6079, 6089, 6091, 6101, 6113, 6121, 6131, 6133,
	6143, 6151, 6163, 6173, 6197, 6199, 6203, 6211,
	6217, 6221, 6229, 6247, 6257, 6263, 6269, 6271,
	6277, 6287, 6299, 6301, 6311, 6317, 6323, 6329,
	6337, 6343, 6353, 6359, 6361, 6367, 6373, 6379,
	6389, 6397, 6421, 6427, 6449, 6451, 6469, 6473,
	6481, 6491, 6521, 6529, 6547, 6551, 6553, 6563,
	6569, 6571, 6577, 6581, 6599, 6607, 6619, 6637,
	6653, 6659, 6661, 6673, 6679, 6689, 6691, 6701,
	6703, 6709, 6719, 6733, 6737, 6761, 6763, 6779,
	6781, 6791, 6793, 6803, 6823, 6827, 6829, 6833,
	6841, 6857, 6863, 6869, 6871, 6883, 6899, 6907,
	6911, 6917, 6947, 6949, 6959, 6961, 6967, 6971,
	6977, 6983, 6991, 6997, 7001, 7013, 7019, 7027,
	7039, 7043, 7057, 7069, 7079, 7103, 7109, 7121,
	7127, 7129, 7151, 7159, 7177, 7187, 7193, 7207,
	7211, 7213, 7219, 7229, 7237, 7243, 7247, 7253,
	7283, 7297, 7307, 7309, 7321, 7331, 7333, 7349,
	7351, 7369, 7393, 7411, 7417, 7433, 7451, 7457,
	7459, 7477, 7481, 7487, 7489, 7499, 7507, 7517,
	7523, 7529, 7537, 7541, 7547, 7549, 7559, 7561,
	7573, 7577, 7583, 7589, 7591, 7603, 7607, 7621,
	7639, 7643, 7649, 7669, 7673, 7681, 7687, 7691,
	7699, 7703, 7717, 7723, 7727, 7741, 7753, 7757,
	7759, 7789, 7793, 7817, 7823, 7829, 7841, 7853,
	7867, 7873, 7877, 7879, 7883, 7901, 7907, 7919,
	7927, 7933, 7937, 7949, 7951, 7963, 7993, 8009,
	8011, 8017, 8039, 8053, 8059, 8069, 8081, 8087,
	8089, 8093, 8101, 8111, 8117, 8123, 8147, 8161,
	8167, 8171, 8179, 8191,
	0}; /* null-terminated list, with only one null at end */




/*
	This routine tests p for primality by applying Fermat's
	theorem: For any x, if (pow(x, (p-1)) mod p) != 1, then
	p is not prime. By trying a few values for x, we can
	determine if p is "probably" prime.

	Because this test is so slow, it is recommended that p
	be sieved first to weed out numbers that are obviously
	not prime.

	Contrary to what you may have read in the literature,
	empirical evidence shows this test weeds out a LOT more
	than 50% of the composite candidates for each trial x.
	Each test catches nearly all the composites.

	Some people have questioned whether four Fermat tests is
	sufficient. See "Finding Four Million Large Random
	Primes", by Ronald Rivest, in Advancess in Cryptology:
	Proceedings of Crypto '91.  He used a small-divisor test
	similar to PGP's, then a Fermat test to the base 2, and
	then 8 iterarions of a Miller-Rabin test.  About 718
	million random 256-bit integers were generated,
	43,741,404 passed the small divisor test, 4,058,000
	passed the Fermat test, and all 4,058,000 passed all 8
	iterations of the Miller-Rabin test, proving their
	primality beyond most reasonable doubts.  This is strong
	experimental evidence that the odds of getting a
	non-prime are less than one in a million (10^-6).

	He also gives a theoretical argument that the chances of
	finding a 256-bit non-prime which satisfies one Fermat
	test to the base 2 is less than 10^-22.  The small
	divisor test improves this number, and if the numbers
	are 512 bits (as needed for a 1024-bit key) the odds of
	failure shrink to about 10^-44.  Thus, he concludes, for
	practical purposes one Fermat test to the base 2 is
	sufficient.
*/
LOCALFUNC tMyErr slowtest(mp_UNIT_p p)
{
	tMyErr err;
	mp_UNIT_t x[MAX_UNIT_PRECISION];
	mp_UNIT_t is_one[MAX_UNIT_PRECISION];
	mp_UNIT_t pminus1[MAX_UNIT_PRECISION];
	short i;

	mp_move(pminus1, p);
	mp_dec(pminus1);

	for (i = 0; i < 4; i++) {        /* Just do a few tests. */
		mp_Ui4rSet(x, primetable[i]); /* Use any old random trial x */
		/* if ((x**(p-1)) mod p) != 1, then p is not prime */

		if (kMyErr_noErr != (err =
			mp_modexp(is_one, x, pminus1, p)))
		{
			goto l_exit;
		}
		if (! mp_isOne(is_one)) {
			/* then p is not prime */
			err = kMyErrNoMatch;
			goto l_exit; /* return not prime status */
		}
#ifdef SHOWPROGRESS
		dbglog_writeChar('*'); /* let user see how we are progressing */
#endif /* SHOWPROGRESS */
	}

	/* If it gets to this point, it's very likely that p is prime */
	err = kMyErr_noErr;

l_exit:

#if DebugCheck
	if ((kMyErr_noErr != err)
		&& (kMyErrNoMatch != err))
	{
		err = ErrReportStack0(err, "slowtest");
	}
#endif

	return err;
}                                /* slowtest -- fermattest */


/*
	Used in conjunction with fastsieve.  Builds a table of
	remainders relative to the random starting point p, so
	that fastsieve can sequentially sieve for suspected
	primes quickly.  Call buildsieve once, then call
	fastsieve for consecutive prime candidates. Note that p
	must be odd, because the sieve begins at 3.
*/
LOCALPROC buildsieve(mp_UNIT_p p, ui4b remainders[])
{
	short i;

	for (i = 1; 0 != primetable[i]; i++) {
		remainders[2 * i] = 0;
		remainders[2 * i + 1] = mp_shortmod(p, primetable[i]);
	}
}                                /* buildsieve */

/*
	Fast prime sieving algorithm by Philip Zimmermann, March
	1987. This is the fastest algorithm I know of for
	quickly sieving for large (hundreds of bits in length)
	"random" probable primes, because it uses only
	single-precision (16-bit) arithmetic.  Because rigorous
	prime testing algorithms are very slow, it is
	recommended that potential prime candidates be quickly
	passed through this fast sieving algorithm first to weed
	out numbers that are obviously not prime.

	This algorithm is optimized to search sequentially for a
	large prime from a random starting point.  For
	generalized nonsequential prime testing, the slower
	conventional sieve should be used, as given in
	primetest(p).

	This algorithm requires a fixed table (called
	primetable) of the first hundred or so small prime
	numbers. First we select a random odd starting point (p)
	for our prime search.  Then we build a table of 16-bit
	remainders calculated from that initial p.  This table
	of 16-bit remainders is exactly the same length as the
	table of small 16-bit primes.  Each remainders table
	entry contains the remainder of p divided by the
	corresponding primetable entry.  Then we begin
	sequentially testing all odd integers, starting from the
	initial odd random p.  The distance we have searched
	from the huge random starting point p is a small 16-bit
	number, pdelta.  If pdelta plus a remainders table entry
	is evenly divisible by the corresponding primetable
	entry, then p+pdelta is factorable by that primetable
	entry, which means p+pdelta is not prime.
*/

/*
	Fastsieve is used for searching sequentially from a
	random starting point for a suspected prime.  Requires
	that buildsieve be called first, to build a table of
	remainders relative to the random starting point p.
	Returns true iff pdelta passes through the sieve and
	thus p+pdelta may be a prime.  Note that p must be odd,
	because the sieve begins at 3.
*/
LOCALFUNC blnr fastsieve(ui4r pdelta, ui4b *remainders, ui4r range,
	ui4r *newdelta)
{
	blnr v;
	ui4r p;
	ui4r r;
	const ui4b *pt0;
	const ui4b *pt = primetable;

	++pt;
	++remainders;
	++remainders;

	pt0 = pt;

l_retry:
	p = *pt++;
	if (0 == p) {
		/* reached end of prime table */
		/* It passed the sieve.  It is now a suspected prime. */
		v = trueblnr;
	} else {
		r = (pdelta - *remainders);
		*remainders++ = pdelta;

		r += *remainders;

		/* r = r % p */
		while (r >= p) {
			r -= p;
		}

		*remainders++ = r;
		/*
			If pdelta plus a remainders table entry is evenly
			divisible by the corresponding primetable entry,
			then p + pdelta is factorable by that primetable
			entry, which means p + pdelta is not prime.
		*/

		if (r != 0) {
			goto l_retry;
		} else {
			/* then p + pdelta is not prime */
#ifndef BLUM
			pdelta += 2;        /* try next odd number */
#else
			pdelta += 4;
#endif
			if (pdelta > range) {
				/* searched too many candidates? */
				/* something must be wrong--bail out of search */
				v = falseblnr;
			} else {
#if 0 /* compiler does not do this very well */
				remainders -= 2 * (pt - pt0);
#else
				remainders = (ui4p) (((ui3p)remainders)
					- 2 * (((ui3p)pt) - ((ui3p)pt0)));
#endif
				pt = pt0;
				goto l_retry;
			}
		}
	}

	*newdelta = pdelta;
	return v;
} /* fastsieve */


#define numberof(x) (sizeof(x)/sizeof(x[0]))
	/* number of table entries */


LOCALFUNC tMyErr nextprime(mp_UNIT_p p)
/*
	Find next higher prime starting at p, returning result in p.
	Uses fast prime sieving algorithm to search sequentially.
	Returns 0 for normal completion status, < 0 for failure status.
*/
{
	tMyErr err;
	ui4r pdelta;
	ui4r range;
	short oldprecision;
	short i;
	short suspects;
	MyHandle remainders;
	ui4r newdelta;

	/* start search at candidate p */
	mp_inc(p);
		/* remember, it's the NEXT prime from p, noninclusive. */
	if (mp_significance(p) <= 1) {
		/*
			p might be smaller than the largest prime in primetable.
			We can't sieve for primes that are already in primetable.
			We will have to directly search the table.
		*/
		/* scan until null-terminator */
		for (i = 0; primetable[i]; i++) {
			if (primetable[i] >= p[0]) {
				mp_Ui4rSet(p, primetable[i]);
				err = kMyErr_noErr;
				goto l_exit;
					/* return next higher prime from primetable */
			}
		} /* We got past the whole primetable without a hit. */
	} /* p is bigger than any prime in primetable, so let's sieve. */

#ifndef BLUM
	p[0] |= 1; /* set candidate's lsb - make it odd */
#else
	p[0] |= 3; /* Make candidate ==3 mod 4 */
#endif

	/*
		Adjust the mp_precision downward to the optimum
		size for p...
	*/
	oldprecision = mp_precision; /* save mp_precision */
	/*
		We will need 2-3 extra bits of precision for the falsekeytest.
	*/
	mp_precision_set(mp_bits2units(mp_countbits(p) + 4 + SLOP_BITS));
	/* rescale p to mp_precision we just defined */

	/* Allocate some memory for the table of remainders: */
	if (kMyErr_noErr != (err =
		MyHandleNew_v2(2 * sizeof(primetable), &remainders)))
	{
		goto l_exit;
	}

	/* Build remainders table relative to initial p: */
	MyHandleLock(remainders);
	buildsieve(p, (ui4b *)MyHandleP(remainders));
	MyHandleUnlock(remainders);
	pdelta = 0;                /* offset from initial random p */
	/*
		Sieve preparation complete.  Now for some fast
		fast sieving...
	*/
	/* slowtest will not be called unless fastsieve is true */

	/* range is how far to search before giving up. */
#ifndef BLUM
	range = 4 * mp_units2bits(mp_precision);
#else
	/* Twice as many because step size is twice as large, */
	range = 8 * mp_units2bits(mp_precision);
#endif
	suspects = 0;
		/* number of suspected primes and slowtest trials */
	MyHandleLock(remainders);
	for (; ; ) {
		/* equivalent to:  if (primetest(p)) break; */
		if (fastsieve(pdelta, (ui4b *)MyHandleP(remainders), range,
			&newdelta))
		{
			mp_Ui4rAdd(p, newdelta - pdelta);
			pdelta = newdelta;
			/* found suspected prime */
			suspects++; /* tally for statistical purposes */
#ifdef SHOWPROGRESS
			dbglog_writeChar('.');
				/* let user see how we are progressing */
#endif                                /* SHOWPROGRESS */
			if (kMyErrNoMatch != (err = slowtest(p))) {
				break; /* found a prime, or error */
			} else {
#ifndef BLUM
				pdelta += 2;        /* try next odd number */
				mp_Ui4rAdd(p, 2);
#else
				pdelta += 4;
				mp_Ui4rAdd(p, 4);
#endif

				if (pdelta > range) {
					/* searched too many candidates? */
					/* something must be wrong--bail out of search */
					err = NOPRIMEFOUND;
					break;
				}
			}
		} else {
			err = NOPRIMEFOUND;
			break; /* did not find a prime */
		}

	} /* while (TRUE) */
	MyHandleUnlock(remainders);

	if (NOPRIMEFOUND == err) {
		if (suspects < 1) {
			/* unreasonable to have found no suspects */
			/* fastsieve failed, probably */
			err = NOSUSPECTS;
		}
	}

#ifdef SHOWPROGRESS
	dbglog_writeChar(' '); /* let user see how we are progressing */
#endif /* SHOWPROGRESS */

	(void) MyHandleDispose_v2(remainders);

	mp_precision_set(oldprecision); /* restore precision */

l_exit:
	return ErrReportStack(err, "nextprime");
}


/*
	We will need a series of truly random bits for key generation.
	In most implementations, our random number supply is derived from
	random keyboard delays rather than a hardware random number
	chip.  So we will have to ensure we have a large enough pool of
	accumulated random numbers from the keyboard.  trueRandAccum()
	performs this operation.
*/

LOCALFUNC int
trueRandByte(void)
{
	tMyErr err;
	ui3r v;

	if (kMyErr_noErr != (err = ReadBuffToByte(&v))) {
		v = -1;
	}

	return v;
}

/* Fills 1 mp_UNIT_t with random bytes, and returns mp_UNIT_t. */
LOCALFUNC mp_UNIT_t randomunit(void)
{
	ui3r i;
	mp_UNIT_t u = 0;

	i = mp_UNIT_sz;
	do {
		u = (u << 8) + trueRandByte();
	} while (--i != 0);

	return u;
}

/*
	Make a random mp_UNIT_t array p with nbits of precision.
	Used mainly to generate large random numbers to search
	for primes.
*/
LOCALPROC randombits(mp_UNIT_p p, short nbits)
{
	mp_SetZero(p);

#if 4 == mp_UNIT_ln2bsz
	/*
		give same results as if 5 == mp_UNIT_ln2bsz
	*/
	while (nbits >= 2 * mp_UNIT_bsz) {
		ui3r v0 = trueRandByte();
		ui3r v1 = trueRandByte();
		ui3r v2 = trueRandByte();
		ui3r v3 = trueRandByte();
		*p++ = (v2 << 8) + v3;
		*p++ = (v0 << 8) + v1;
		nbits -= 2 * mp_UNIT_bsz;
	}
	if (0 != nbits) {
#if DebugCheck
		dbglog_writeln("randombits nbits not even multiple");
#endif
	}
#endif
#if 5 == mp_UNIT_ln2bsz
	/* Add whole units of randomness */
	while (nbits >= mp_UNIT_bsz) {
		*p++ = randomunit();
		nbits -= mp_UNIT_bsz;
	}

	/* Add most-significant partial mp_UNIT_t (if any) */
	if (0 != nbits) {
		*p = randomunit() & (power_of_2(nbits) - 1);
	}
#endif
}                                /* randombits */

/*
	Makes a "random" prime p with nbits significant bits of
	precision. Since these primes are used to compute a
	modulus of a guaranteed length, the top 2 bits of the
	prime are set to 1, so that the product of 2 primes (the
	modulus) is of a deterministic length. Returns 0 for
	normal completion status, < 0 for failure status.
*/
LOCALFUNC tMyErr randomprime(mp_UNIT_p p, short nbits)
{
	tMyErr err;

	/* Get an initial random candidate p to start search. */
	randombits(p, nbits /* - 2 */);
		/* 2 less random bits for nonrandom top bits */
	/*
		To guarantee exactly nbits of mp_significance,
		set the top 2 bits to 1
	*/
	mp_setbit(p, nbits - 1); /* highest bit is nonrandom */
	mp_setbit(p, nbits - 2); /* next highest bit is also nonrandom */

	/*
		search for next higher prime
		from starting point p
	*/
	if (kMyErr_noErr != (err =
		nextprime(p)))
	{
		/* fail */
	} else
	if (mp_countbits(p) != nbits)
	{
		/*
			overflow? not likely in the entire history of
			the universe, but not actually impossible.
		*/
		err = kMyErr_failr;
	} else
	{
		err = kMyErr_noErr;
	}

	return ErrReportStack(err, "randomprime");
}

LOCALFUNC tMyErr two_random_primes(MyHandle random_H,
	mp_UNIT_p p, mp_UNIT_p q, ui4r pbits, ui4r qbits)
{
	tMyErr err;

	if (kMyErr_noErr != (err =
		ReadFromBuffXhndBegin(random_H)))
	{
		/* fail */
	} else
	{
		if (kMyErr_noErr != (err =
			randomprime(p, pbits)))
		{
			/* fail */
		} else
		if (kMyErr_noErr != (err =
			randomprime(q, qbits)))
		{
			/* fail */
		} else
		{
			err = kMyErr_noErr;
		}

		err = ReadBuffFromXhndEnd(err);
	}

	return ErrReportStack(err, "two_random_primes");
}

/* Define some error status returns for RSA keygen... */
#define KEYFAILED -15 /* key failed final test */


#define swap(p, q) { \
	mp_UNIT_p t; \
	t = p; \
	p = q; \
	q = t; \
}


LOCALPROC derive_rsakeys(mp_UNIT_p n, mp_UNIT_p e, mp_UNIT_p d,
	mp_UNIT_p p, mp_UNIT_p q, mp_UNIT_p u, short ebits)
/*
	Given primes p and q, derive RSA key components n, e, d,
	and u. The mp_precision must have already been set
	large enough for n. Note that p must be < q.
	Primes p and q must have been previously generated
	elsewhere. The bit precision of e will be >= ebits.  The
	search for a usable exponent e will begin with an
	ebits-sized number.  The recommended value for ebits is
	5, for efficiency's sake.  This could yield an e as
	small as 17.
*/
{
	mp_UNIT_t F[MAX_UNIT_PRECISION];
	mp_UNIT_p ptemp, qtemp, phi, G; /* scratchpads */

	/*
		For strong prime generation only, latitude is the amount
		which the modulus may differ from the desired bit precision.
		It must be big enough to allow primes to be generated by
		goodprime reasonably fast.
	*/
#define latitude(bits) (max(min(bits / 16, 12), 4))
	/* between 4 and 12 bits */

	ptemp = d; /* use d for temporary scratchpad array */
	qtemp = u; /* use u for temporary scratchpad array */
	phi = n; /* use n for temporary scratchpad array */
	G = F; /* use F for both G and F */

	if (mp_ge(p, q)) { /* ensure that p<q for computing u */
		swap(p, q); /* swap the pointers p and q */
	}

	/*
		phi(n) is the Euler totient function of n, or the number of
		positive integers less than n that are relatively prime to n.
		G is the number of "spare key sets" for a given modulus n.
		The smaller G is, the better.  The smallest G can get is 2.
	*/
	mp_move(ptemp, p);
	mp_move(qtemp, q);
	mp_dec(ptemp);
	mp_dec(qtemp);
	mp_mult(phi, ptemp, qtemp); /*  phi(n) = (p-1)*(q-1)  */
	mp_gcd(G, ptemp, qtemp); /*  G(n) = gcd(p-1,q-1)  */
	mp_udiv(ptemp, qtemp, phi, G); /* F(n) = phi(n)/G(n)  */
	mp_move(F, qtemp);

	/*
		We now have phi and F.  Next, compute e...
		Strictly speaking, we might get slightly faster results by
		testing all small prime e's greater than 2 until we hit a
		good e.  But we can do just about as well by testing all
		odd e's greater than 2.
		We could begin searching for a candidate e anywhere, perhaps
		using a random 16-bit starting point value for e, or even
		larger values.  But the most efficient value for e would be 3,
		if it satisfied the gcd test with phi.
		Parameter ebits specifies the number of significant bits e
		should have to begin search for a workable e.
		Make e at least 2 bits long, and no longer than one bit
		shorter than the length of phi.
	*/
	ebits = min(ebits, mp_countbits(phi) - 1);
	if (ebits == 0) {
		ebits = 5; /* default is 5 bits long */
	}
	ebits = max(ebits, 2);
	mp_SetZero(e);
	mp_setbit(e, ebits - 1);
	e[0] |= 1; /* set e candidate's lsb - make it odd */
	mp_Ui4rSub(e, 2); /* precompensate for preincrements of e */
	do {
		mp_Ui4rAdd(e, 2); /* try odd e's until we get it. */
		mp_gcd(ptemp, e, phi);
			/* look for e such that gcd(e, phi(n)) = 1 */
	} while (! mp_isOne(ptemp));

	/*
		Now we have e.  Next, compute d, then u, then n.
		d is the multiplicative inverse of e, mod F(n).
		u is the multiplicative inverse of p, mod q, if p<q.
		n is the public modulus p*q.
	*/
	mp_inv(d, e, F); /* compute d such that (e*d) mod F(n) = 1 */
	mp_inv(u, p, q); /* (p*u) mod q = 1, assuming p<q */
	mp_mult(n, p, q); /*  n = p*q  */
}

LOCALFUNC tMyErr rsa_keygencheck(
	mp_UNIT_p n, mp_UNIT_p e, mp_UNIT_p d,
	mp_UNIT_p p, mp_UNIT_p q, mp_UNIT_p u)
{
	tMyErr err;
	ui3b a0[MAX_BYTE_PRECISION];
	mp_UNIT_t temp0[MAX_UNIT_PRECISION];
	mp_UNIT_t temp1[MAX_UNIT_PRECISION];
	mp_UNIT_t C[MAX_UNIT_PRECISION];
	mp_UNIT_t t[MAX_UNIT_PRECISION];
	mp_UNIT_t DP[MAX_UNIT_PRECISION], DQ[MAX_UNIT_PRECISION];
	ui4r bytes = 16;
	uimr nUnits = bytes >> mp_UNIT_ln2sz;
	int i;

	if (bytes != nUnits << mp_UNIT_ln2sz) {
		err = -9;
		goto l_exit;
	}

	if (nUnits >= mp_significance(n)) {
		err = -9;
		goto l_exit;
	}

	/* Create a dummy signature */

	for (i = 0; i < bytes; i++) {
		a0[i] = 3 * i + 7;
	}

	{
		ui4r save_precision = mp_precision;

		mp_precision_set(nUnits);

		mp_converfromLSB(temp0, a0);

		mp_precision_set(save_precision);

		mp_SetRestZero(temp0, nUnits);
	}

	mp_move(t, p);
	mp_dec(t);
	mp_mod(DP, d, t);
	mp_move(t, q);
	mp_dec(t);
	mp_mod(DQ, d, t);

	if (kMyErr_noErr != (err =
		mp_modexp_crt(C, temp0, p, q, DP, DQ, u)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		mp_modexp(temp1, C, e, n)))
	{
		goto l_exit;
	}

	if (! mp_eq(temp0, temp1)) {
		err = KEYFAILED;
		goto l_exit;
	}

	err = kMyErr_noErr;

l_exit:
	return err;
}

LOCALFUNC tMyErr rsa_keygen0(MyHandle random_H,
	mp_UNIT_p n, mp_UNIT_p e, mp_UNIT_p d,
	mp_UNIT_p p, mp_UNIT_p q, mp_UNIT_p u, short keybits, short ebits)
/*
	single attempt at key generation
	if fail, try again with different random data,
		at least the first few times.
*/
/*
	Generate RSA key components p, q, n, e, d, and u.
	This routine sets the mp_precision appropriate for n,
	where keybits is desired precision of modulus n.
	The precision of exponent e will be >= ebits.
	It will generate a p that is < q.
	Returns 0 for succcessful keygen, negative status otherwise.
*/
{
	tMyErr err;
	short pbits, qbits;
	int slop;

	/*
		Don't let keybits get any smaller than 2 units, because
		some parts of the math package require at least 2 units
		for mp_precision.
		Nor any smaller than the 32 bits of preblocking overhead.
		Nor any bigger than MAX_BIT_PRECISION - SLOP_BITS.
		Also, if generating "strong" primes, don't let keybits get
		any smaller than 64 bits, because of the search latitude.
	*/
	slop = max(SLOP_BITS, 1);
		/* allow at least 1 slop bit for sign bit */
	keybits = min(keybits, (MAX_BIT_PRECISION - slop));
	keybits = max(keybits, mp_UNIT_bsz * 2);
	keybits = max(keybits, 32); /* minimum preblocking overhead */

	mp_precision_set(mp_bits2units(keybits + slop));

	/*
		We will need a series of truly random bits to generate the
		primes.  We need enough random bits for keybits, plus two
		random units for combined discarded bit losses in randombits.
		Since we now know how many random bits we will need,
		this is the place to prefill the pool of random bits.
	*/

#if 0
	/*
		If you want primes of different lengths
		("separation" bits apart), do the following:
	*/
	pbits = (keybits - separation) / 2;
#else
	pbits = keybits / 2;
#endif
	qbits = keybits - pbits;

	/*
		This load of random bits will be stirred and recycled until
		a good q is generated.
	*/

	if (kMyErr_noErr != (err =
		two_random_primes(random_H, p, q, pbits, qbits)))
	{
		goto l_exit;
	}

	/* Note that at this point we can't be sure that q>p. */
	if (mp_ge(p, q)) {
		/* ensure that p<q for computing u */
		mp_move(u, p);
		mp_move(p, q);
		mp_move(q, u);
	}

	/* See if p and q are far enough apart.  Is q-p big enough? */
	mp_move(u, q); /* use u as scratchpad */
	mp_sub(u, p); /* compute q-p */
	if (mp_countbits(u) < (mp_countbits(q) - 7)) {
		/* too close together */

		return -1;
		/* Keep trying q's until we get one far enough from p... */
	}

	derive_rsakeys(n, e, d, p, q, u, ebits);

	mp_precision_set(mp_bits2units(keybits));
		/* no slop */

	/* Now test key just to make sure --this had better work! */
	err = rsa_keygencheck(n, e, d, p, q, u);

l_exit:
	return err;
}

LOCALFUNC tMyErr rsa_keygen(MyHandle random_H,
	mp_UNIT_p n, mp_UNIT_p e, mp_UNIT_p d,
	mp_UNIT_p p, mp_UNIT_p q, mp_UNIT_p u, short keybits, short ebits)
{
	tMyErr err;
	uimr retry_count = 0;

l_retry:
	if (kMyErr_noErr != (err =
		rsa_keygen0(random_H, n, e, d, p, q, u, keybits, ebits)))
	{
		++retry_count;
		if ((kMyErrUsrAbort != err) && (retry_count < 3)) {
			if (kMyErr_noErr == (err =
				MyHandRotateLeft(random_H)))
			{
				goto l_retry;
			}
		}
	}

	return ErrReportStack(err, "rsa_keygen");
}

LOCALFUNC tMyErr rsa_keygen1(MyHandle random_H,
	mp_UNIT_p n, mp_UNIT_p e, mp_UNIT_p d,
	mp_UNIT_p p, mp_UNIT_p q, mp_UNIT_p u)
{
	tMyErr err;
	uimr L;
	short keybits;
	short ebits;

	ebits = 17;

	if (kMyErr_noErr == (err =
		MyHandleGetSize_v2(random_H, &L)))
	{
		keybits = ((L * 8) / 128) * 128;

		if (keybits < 384) {
			err = -1;
		} else
		{
			err = rsa_keygen(random_H, n, e, d, p, q, u,
				keybits, ebits);
		}
	}

	return ErrReportStack(err, "rsa_keygen1");
}

#define Have_PKGNKEYS 1
