/*
	PRFXCODE.h

	Copyright (c) 2017-present, MacPaw Inc., Paul C. Pratt

	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.

	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	Lesser General Public License for more details.

	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
	MA 02110-1301  USA
*/

/*
	PReFiX CODE
*/

typedef struct XADCodeTreeNode XADCodeTreeNode;
typedef struct XADCodeTableEntry XADCodeTableEntry;

struct XADPrefixCode
{
	XADCodeTreeNode *tree;
	ui5r numentries;
	ui3r minlength;
	ui3r maxlength;

	ui3r tablesize;
	XADCodeTableEntry *table2;
};
typedef struct XADPrefixCode XADPrefixCode;



struct XADCodeTreeNode
{
	si5r branches[2];
};

struct XADCodeTableEntry
{
	ui3r length;
	si5r value;
};


static inline XADCodeTreeNode *NodePointer(XADPrefixCode *self,
	si5r node)
{
	return &self->tree[node];
}

static inline si5r Branch(XADPrefixCode *self, si5r node, ui3r bit)
{
	return NodePointer(self, node)->branches[bit];
}

static inline void SetBranch(XADPrefixCode *self,
	si5r node, ui3r bit, si5r nextnode)
{
	NodePointer(self, node)->branches[bit] = nextnode;
}

static inline si5r LeftBranch(XADPrefixCode *self, si5r node)
{
	return Branch(self, node, 0);
}

static inline si5r RightBranch(XADPrefixCode *self, si5r node)
{
	return Branch(self, node, 1);
}

static inline void SetLeftBranch(XADPrefixCode *self,
	si5r node, si5r nextnode)
{
	SetBranch(self, node, 0, nextnode);
}

static inline void SetRightBranch(XADPrefixCode *self,
	si5r node, si5r nextnode)
{
	SetBranch(self, node, 1, nextnode);
}

static inline int LeafValue(XADPrefixCode *self, si5r node)
{
	return LeftBranch(self, node);
}

static inline void SetLeafValue(XADPrefixCode *self,
	si5r node, si5r value)
{
	SetLeftBranch(self, node, value);
	SetRightBranch(self, node, value);
}

static inline void SetEmptyNode(XADPrefixCode *self, si5r node)
{
	SetLeftBranch(self, node, -1);
	SetRightBranch(self, node, -2);
}

static inline blnr IsInvalidNode(si5r node)
{
	return node < 0;
}

static inline blnr IsOpenBranch(XADPrefixCode *self,
	si5r node, ui3r bit)
{
	return IsInvalidNode(Branch(self, node, bit));
}

static inline blnr IsEmptyNode(XADPrefixCode *self, si5r node)
{
	return (LeftBranch(self, node) == -1)
		&& (RightBranch(self, node) == -2);
}

static inline blnr IsLeafNode(XADPrefixCode *self, si5r node)
{
	return LeftBranch(self, node) == RightBranch(self, node);
}

static void MakeTableLE(XADPrefixCode *code,
	si5r node, XADCodeTableEntry *table, ui3r depth, ui3r maxdepth)
{
	ui5r i;
	ui5r currtablesize = 1 << (maxdepth - depth);
	ui5r currstride = 1 << depth;

	if (IsInvalidNode(node)) {
		for (i = 0; i < currtablesize; i++) {
			table[i * currstride].length = -1;
		}
	} else
	if (IsLeafNode(code, node)) {
		for (i = 0; i < currtablesize; i++) {
			table[i * currstride].length = depth;
			table[i * currstride].value = LeafValue(code, node);
		}
	} else
	{
		if (depth == maxdepth) {
			table[0].length = maxdepth + 1;
			table[0].value = node;
		} else
		{
			MakeTableLE(code, LeftBranch(code, node),
				table, depth + 1, maxdepth);
			MakeTableLE(code, RightBranch(code, node),
				table + currstride, depth + 1, maxdepth);
		}
	}
}

#define TableMaxSize 10

static tMyErr _makeTableLE0(XADPrefixCode *self)
{
	tMyErr err;

	if (self->table2) {
		err = kMyErr_noErr;
		goto l_exit;
	}

	if (self->maxlength < self->minlength) {
		self->tablesize = TableMaxSize; /* no code lengths recorded */
	} else if (self->maxlength >= TableMaxSize) {
		self->tablesize = TableMaxSize;
	} else {
		self->tablesize = self->maxlength;
	}

	if (NULL == (self->table2 = malloc(sizeof(XADCodeTableEntry)
		* (1 << self->tablesize))))
	{
		err = errno2MyErr();
	} else
	{
		MakeTableLE(self, 0, self->table2, 0, self->tablesize);

		err = kMyErr_noErr;
	}

l_exit:
	return err;
}

static inline tMyErr NewNode(XADPrefixCode *self, ui5r *x)
{
	tMyErr err;

	self->tree = realloc(self->tree,
		(self->numentries + 1) * sizeof(XADCodeTreeNode));
	if (! self->tree) {
		err = kMyErr_lowmem;
	} else {
		SetEmptyNode(self, self->numentries);
		*x = self->numentries++;
		err = kMyErr_noErr;
	}

	return err;
}


static tMyErr CSInputNextSymbolUsingCodeLE(XADPrefixCode *code, ui5r *x)
{
	tMyErr err;
	ui5r bits;
	int length;
	int value;

	if (! code->table2) {
		if (kMyErr_noErr != (err = _makeTableLE0(code))) {
			goto l_exit;
		}
	}

	if (kMyErr_noErr != (err =
		CSInputPeekBitStringLE(code->tablesize, &bits)))
	{
		goto l_exit;
	}

	length = code->table2[bits].length;
	value = code->table2[bits].value;

	if (length < 0) {
		err = kMyErr_crptfl;
		goto l_exit;
	}

	if (length <= code->tablesize) {
		if (kMyErr_noErr != (err = CSInputSkipPeekedBitsLE(length))) {
			goto l_exit;
		}

		*x = value;
		err = kMyErr_noErr;
	} else {
		si5r node;

		if (kMyErr_noErr != (err =
			CSInputSkipPeekedBitsLE(code->tablesize)))
		{
			goto l_exit;
		}

		node = value;
		while (! IsLeafNode(code, node)) {
			ui3r bit;

			if (kMyErr_noErr != (err = CSInputNextBitLE(&bit))) {
				goto l_exit;
			}

			if (IsOpenBranch(code, node, bit)) {
				err = kMyErr_crptfl;
				/* report an eror : XADInvalidPrefixCodeException*/
				/* "Invalid prefix code in bitstream" */
				goto l_exit;
			}

			node = Branch(code, node, bit);
		}

		*x = LeafValue(code, node);
		err = kMyErr_noErr;
	}

l_exit:
	return err;
}

static void XADPrefixCodeFree(XADPrefixCode **code)
{
	XADPrefixCode *x = *code;

	if (NULL != x) {
		*code = NULL;

		if (NULL != x->tree) {
			free(x->tree);
		}
		if (NULL != x->table2) {
			free(x->table2);
		}
		free(x);
	}
}

static tMyErr XADPrefixCodeNewEmpty(XADPrefixCode **x)
{
	tMyErr err;
	XADPrefixCode *r;

	if (NULL == (r = malloc(sizeof(XADPrefixCode)))) {
		err = errno2MyErr();
	} else
	{
		r->numentries = 1;
		r->minlength = 0xFF;
		r->maxlength = 0;

		r->tree = NULL;
		r->table2 = NULL;

		*x = r;

		if (NULL == (r->tree = malloc(sizeof(XADCodeTreeNode)))) {
			err = errno2MyErr();
		} else {
			SetEmptyNode(r, 0);
			err = kMyErr_noErr;
		}
	}

	return err;
}

static tMyErr XADPrefixCodeAddValueHighBitFirst(XADPrefixCode *self,
	int value, ui5r code, ui3r length)
{
	tMyErr err;
	int bitpos;
	int lastnode;

	if (NULL != self->table2) {
		free(self->table2);
		self->table2 = NULL;
	}

	if (length>self->maxlength) {
		self->maxlength = length;
	}
	if (length<self->minlength) {
		self->minlength = length;
	}

	lastnode = 0;
	for (bitpos = length - 1; bitpos >= 0; bitpos--) {
		ui3r bit = (code >> bitpos) & 1;

		if (IsLeafNode(self, lastnode)) {
			/* "Prefix found" */
			err = kMyErr_crptfl;
			goto l_exit;
		}

		if (IsOpenBranch(self, lastnode, bit)) {
			ui5r x;

			if (kMyErr_noErr != (err = NewNode(self, &x))) {
				goto l_exit;
			}

			SetBranch(self, lastnode, bit, x);
		}
		lastnode = Branch(self, lastnode, bit);
	}

	if (! IsEmptyNode(self, lastnode)) {
		/* "Prefix found" */
		err = kMyErr_crptfl;
		goto l_exit;
	}
	SetLeafValue(self, lastnode, value);

	err = kMyErr_noErr;

l_exit:
	return err;
}

static ui5r Reverse32(ui5r val)
{
	val = ((val >> 1) & 0x55555555) | ((val & 0x55555555) << 1);
	val = ((val >> 2) & 0x33333333) | ((val & 0x33333333) << 2);
	val = ((val >> 4) & 0x0F0F0F0F) | ((val & 0x0F0F0F0F) << 4);
	val = ((val >> 8) & 0x00FF00FF) | ((val & 0x00FF00FF) << 8);
	return (val >> 16) | (val << 16);
}

static ui5r ReverseN(ui5r val, ui3r length)
{
	return Reverse32(val) >> (32 - length);
}

static tMyErr XADPrefixCodeAddValueLowBitFirst(XADPrefixCode *self,
	int value, ui5r code, ui3r length)
{
	return XADPrefixCodeAddValueHighBitFirst(self, value,
		ReverseN(code, length), length);
}

static tMyErr XADPrefixCodeNewWithLengths(const ui3b *lengths,
	ui5r numsymbols, ui3r maxcodelength, XADPrefixCode **x)
{
	tMyErr err;
	int length;
	int i;

	if (kMyErr_noErr != (err = XADPrefixCodeNewEmpty(x))) {
		/* fail */
	} else {
		ui5r code = 0;
		int symbolsleft = numsymbols;
		XADPrefixCode *r = *x;

		for (length = 1; length <= maxcodelength; length++) {
			for (i = 0; i < numsymbols; i++) {
				if (lengths[i] != length) {
					continue;
				}
				/*
					Instead of reversing to get a low-bit-first code,
					we shift and use high-bit-first.
				*/
				if (kMyErr_noErr != (err =
					XADPrefixCodeAddValueHighBitFirst(r,
						i, code, length)))
				{
					goto l_exit;
				}

				code++;
				if (0 == --symbolsleft) {
					/* early exit if all codes have been handled */
					err = kMyErr_noErr;
					goto l_exit;
				}
			}
			code <<= 1;
		}

		err = kMyErr_noErr;
	}

l_exit:
	return err;
}
