/*
	RDWRFILE.h

	Copyright (c) 2017-present, MacPaw Inc., Paul C. Pratt

	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.

	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	Lesser General Public License for more details.

	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
	MA 02110-1301  USA
*/

/*
	ReaD and WRite FILEs
*/


static tMyErr syserr2MyErr(int e)
{
	/* assuming an error occurred */
	tMyErr err;

	switch (e) {
#ifdef ENOSPC
		case ENOSPC:
			err = kMyErr_lowdsk;
			break;
#endif
#ifdef ENOMEM
		case ENOMEM:
			err = kMyErr_lowmem;
			break;
#endif
#ifdef ENOENT
		case ENOENT:
			err = kMyErr_flnfnd;
			break;
#endif
		default:
#if DebugCheck
			dbglog_writelnNum("unknown error", e);
#endif
			err = kMyErrSysUnknown;
			break;
	}

	return ErrReportStack(err, "syserr2MyErr");
}

static tMyErr errno2MyErr(void)
{
	/* assuming an error occurred */
	return syserr2MyErr(errno);
}

static tMyErr ferror2MyErr(FILE *in_fh)
{
	/* assuming an error occurred */
	return syserr2MyErr(ferror(in_fh));
}

static inline ui4r CSUInt16BE(const ui3b *b)
{
	return ((ui4r)b[0] << 8) | (ui4r)b[1];
}

static inline ui5r CSUInt32BE(const ui3b *b)
{
	return ((ui5r)b[0] << 24)
		| ((ui5r)b[1] << 16)
		| ((ui5r)b[2] << 8)
		| (ui5r)b[3];
}

static FILE *in_fh = NULL;

static void CSFileInFree(void)
{
	if (NULL != in_fh) {
		fclose(in_fh);
		in_fh = NULL;
	}
}

static tMyErr CSFileInOpen(char *path)
{
	tMyErr err;

	if (NULL != in_fh) {
		err = kMyErrProgramBug;
	} else
	if (NULL == (in_fh = fopen(path, "rb"))) {
		err = errno2MyErr();
	} else
	{
		err = kMyErr_noErr;
	}

	return ErrReportStack(err, "CSFileInOpen");
}

static tMyErr CSFileInReadBytes(uimr num, void *buffer)
{
	tMyErr err;

	if (0 == num) {
		err = kMyErr_noErr;
	} else
	if (num != fread(buffer, 1, num, in_fh)) {
		err = ferror2MyErr(in_fh);
	} else
	{
		err = kMyErr_noErr;
	}

	return ErrReportStack(err, "CSFileInReadBytes");
}

static tMyErr CSFileInReadByte(ui3r *v)
{
	tMyErr err;
	ui3b buf;

	if (kMyErr_noErr != (err = CSFileInReadBytes(1, &buf))) {
		/* fail */
	} else {
		err = kMyErr_noErr;
		*v = buf;
	}

	return ErrReportStack(err, "CSFileInReadByte");
}

static tMyErr CSFileInReadUi4BE(ui4r *v)
{
	tMyErr err;
	ui3r b0;
	ui3r b1;

	if (kMyErr_noErr != (err = CSFileInReadByte(&b0))) {
		/* fail */
	} else
	if (kMyErr_noErr != (err = CSFileInReadByte(&b1))) {
		/* fail */
	} else
	{
		*v = ((ui4r)b1)
			| (((ui4r)b0) << 8);
		err = kMyErr_noErr;
	}

	return ErrReportStack(err, "CSFileInReadUi4BE");
}

static tMyErr CSFileInReadUi5BE(ui5r *v)
{
	tMyErr err;
	ui4r b0;
	ui4r b1;

	if (kMyErr_noErr != (err = CSFileInReadUi4BE(&b0))) {
		/* fail */
	} else
	if (kMyErr_noErr != (err = CSFileInReadUi4BE(&b1))) {
		/* fail */
	} else
	{
		*v = ((ui5r)b1)
			| (((ui5r)b0) << 16);
		err = kMyErr_noErr;
	}

	return ErrReportStack(err, "CSFileInReadUi5BE");
}

static tMyErr CSFileInSkipBytes(ui5r n)
{
	tMyErr err;
	ui3r b;

l_retry:
	if (0 == n) {
		err = kMyErr_noErr;
	} else
	if (kMyErr_noErr != (err = CSFileInReadByte(&b))) {
		/* fail */
	} else
	{
		--n;
		goto l_retry;
	}

	return ErrReportStack(err, "CSFileInSkipBytes");
}

static FILE *out_fh = NULL;

static void CSFileOutFree(void)
{
	if (NULL != out_fh) {
		fclose(out_fh);
		out_fh = NULL;
	}
}

static tMyErr CSFileOutOpen(char *path)
{
	tMyErr err;

	if (NULL != out_fh) {
		err = kMyErrProgramBug;
	} else
	if (NULL == (out_fh = fopen(path, "wb"))) {
		err = errno2MyErr();
	} else
	{
		err = kMyErr_noErr;
	}

	return ErrReportStack(err, "CSFileOutOpen");
}

static tMyErr CSFileOutWriteByte(ui3r v)
{
	tMyErr err;

	if (fwrite(&v, 1, 1, out_fh) != 1) {
		err = ferror2MyErr(out_fh);
	} else {
		err = kMyErr_noErr;
	}

	return ErrReportStack(err, "CSFileOutWriteByte");
}
