/*
	STFFIT13.h

	Copyright (c) 2017-present, MacPaw Inc., Paul C. Pratt

	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License as published by the Free Software Foundation; either
	version 2.1 of the License, or (at your option) any later version.

	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	Lesser General Public License for more details.

	You should have received a copy of the GNU Lesser General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
	MA 02110-1301  USA
*/

/*
	STuFFIT 13
*/


static const ui5r MetaCodes[37];
static const ui3r MetaCodeLengths[37];

static XADPrefixCode *firstcode = NULL;
static XADPrefixCode *secondcode = NULL;
static XADPrefixCode *offsetcode = NULL;
static XADPrefixCode *currcode = NULL;

#define MaxNumCodes 512

static tMyErr allocAndParseCodeOfSize(ui4r numcodes,
	XADPrefixCode *metacode, XADPrefixCode **r)
{
	tMyErr err;
	ui5r x;
	ui4r i;
	ui3r length = 0;
	ui3b lengths[MaxNumCodes];

	for (i = 0; i < numcodes; i++) {
		ui5r val;

		if (kMyErr_noErr != (err =
			CSInputNextSymbolUsingCodeLE(metacode, &val)))
		{
			goto l_exit;
		}

		switch (val) {
			case 31:
				length = -1;
				break;
			case 32:
				length++;
				break;
			case 33:
				length--;
				break;
			case 34:
				{
					ui3r b;

					if (kMyErr_noErr != (err = CSInputNextBitLE(&b))) {
						goto l_exit;
					}

					if (0 != b) {
						lengths[i++] = length;
					}
				}
				break;
			case 35:
				if (kMyErr_noErr != (err =
					CSInputNextBitStringLE(3, &x)))
				{
					goto l_exit;
				}

				val = x + 2;
				while (val--) {
					lengths[i++] = length;
				}
				break;
			case 36:
				if (kMyErr_noErr != (err =
					CSInputNextBitStringLE(6, &x)))
				{
					goto l_exit;
				}

				val = x + 10;
				while (val--) {
					lengths[i++] = length;
				}
				break;
			default:
				length = val + 1;
				break;
		}
		lengths[i] = length;
	}

	err = XADPrefixCodeNewWithLengths(lengths, numcodes, 32, r);

l_exit:
	return err;
}

static tMyErr resetLZSSHandle(ui3r val)
{
	tMyErr err;
	ui3r i;
	XADPrefixCode *metacode = NULL;

	if (0 != (val >> 4)) {
		err = kMyErrParamErr;
		goto l_exit;
	} else
	if (0 != (val & 0x08)) {
		err = kMyErrParamErr;
		goto l_exit;
	} else
	{
		if (kMyErr_noErr != (err = XADPrefixCodeNewEmpty(&metacode))) {
			goto l_exit;
		}

		for (i = 0; i < 37; i++) {
			if (kMyErr_noErr != (err =
				XADPrefixCodeAddValueLowBitFirst(metacode, i,
					MetaCodes[i], MetaCodeLengths[i])))
			{
				goto l_exit;
			}
		}

		if (kMyErr_noErr != (err =
			allocAndParseCodeOfSize(321, metacode, &firstcode)))
		{
			goto l_exit;
		}

		if (kMyErr_noErr != (err =
			allocAndParseCodeOfSize(321, metacode, &secondcode)))
		{
			goto l_exit;
		}

		if (kMyErr_noErr != (err =
			allocAndParseCodeOfSize((val & 0x07) + 10, metacode,
				&offsetcode)))
		{
			goto l_exit;
		}

		currcode = firstcode;

		err = kMyErr_noErr;
	}

l_exit:
	XADPrefixCodeFree(&metacode);

	return err;
}

static tMyErr XADStuffIt13ReadAndWrite(ui5r dst_length)
{
	tMyErr err;
	ui3r v0;
	ui5r windowmask;
	ui5r bytesproduced;
	ui5r windowsize = 65536;
	ui3b *windowbuffer = NULL;

	if (kMyErr_noErr != (err = CSFileInReadByte(&v0))) {
		goto l_exit;
	}

	if (kMyErr_noErr != (err = CSInputBufferAlloc())) {
		goto l_exit;
	}

	if (NULL == (windowbuffer = malloc(windowsize))) {
		err = errno2MyErr();
		goto l_exit;
	}

	windowmask = windowsize - 1;
		/* Assumes windows are always power-of-two sized! */

	memset(windowbuffer, 0, windowsize);

	if (kMyErr_noErr != (err = resetLZSSHandle(v0))) {
		goto l_exit;
	}

	bytesproduced = 0;

	while (bytesproduced < dst_length) {
		ui5r x;
		ui5r offset;
		ui5r length;
		ui5r val;
		ui3r byte;

		if (kMyErr_noErr !=
			CSInputNextSymbolUsingCodeLE(currcode, &val))
		{
			goto l_exit;
		}

		if (val < 0x100) {
			currcode = firstcode;

			byte = val;

			windowbuffer[bytesproduced & windowmask] = byte;
			if (kMyErr_noErr != (err = CSFileOutWriteByte(byte))) {
				goto l_exit;
			}

			bytesproduced++;
		} else {
			ui5r bitlength;
			ui5r matchoffset;

			currcode = secondcode;

			if (val < 0x13e) {
				length = val - 0x100 + 3;
			} else if (val == 0x13e) {
				if (kMyErr_noErr != (err =
					CSInputNextBitStringLE(10, &x)))
				{
					goto l_exit;
				}

				length = x + 65;
			} else if (val == 0x13f) {
				if (kMyErr_noErr != (err =
					CSInputNextBitStringLE(15, &x)))
				{
					goto l_exit;
				}

				length = x + 65;
			} else {
				err = kMyErrParamErr;
				goto l_exit;
			}

			if (kMyErr_noErr != (err =
				CSInputNextSymbolUsingCodeLE(offsetcode, &bitlength)))
			{
				goto l_exit;
			}

			if (0 == bitlength) {
				offset = 1;
			} else
			if (1 == bitlength) {
				offset = 2;
			} else
			{
				if (kMyErr_noErr != (err =
					CSInputNextBitStringLE(bitlength - 1, &x)))
				{
					goto l_exit;
				}

				offset = (1 << (bitlength - 1))
					+ x + 1;
			}

			matchoffset = bytesproduced - offset;

			while (0 != length) {
				byte = windowbuffer[matchoffset++ & windowmask];

				windowbuffer[bytesproduced & windowmask] = byte;
				if (kMyErr_noErr != (err = CSFileOutWriteByte(byte)))
				{
					goto l_exit;
				}
				++bytesproduced;
				--length;
			}
		}
	}


	err = kMyErr_noErr;

l_exit:
	XADPrefixCodeFree(&firstcode);
	XADPrefixCodeFree(&secondcode);
	XADPrefixCodeFree(&offsetcode);
	if (NULL != windowbuffer) {
		free(windowbuffer);
	}
	CSInputBufferFree();

	return err;
}


static const ui5r MetaCodes[37]=
{
	0x5d8, 0x058, 0x040, 0x0c0, 0x000, 0x078, 0x02b, 0x014,
	0x00c, 0x01c, 0x01b, 0x00b, 0x010, 0x020, 0x038, 0x018,
	0x0d8, 0xbd8, 0x180, 0x680, 0x380, 0xf80, 0x780, 0x480,
	0x080, 0x280, 0x3d8, 0xfd8, 0x7d8, 0x9d8, 0x1d8, 0x004,
	0x001, 0x002, 0x007, 0x003, 0x008
};

static const ui3r MetaCodeLengths[37]=
{
	11, 8, 8, 8, 8, 7, 6, 5, 5, 5, 5, 6, 5, 6, 7, 7, 9, 12, 10, 11, 11,
	12, 12, 11, 11, 11, 12, 12, 12, 12, 12, 5, 2, 2, 3, 4, 5
};
