/*
	PKGRYLIB.h
	Copyright (C) 2018 Paul C. Pratt

	You can redistribute this file and/or modify it under the terms
	of version 2 of the GNU General Public License as published by
	the Free Software Foundation.  You should have received a copy
	of the license along with this file; see the file COPYING.

	This file is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	license for more details.
*/

/*
	Public Key signatures (GRYphel format) LIBrary
*/

#define gry_KEYFRAGSIZE 9 /* # of bytes in key ID modulus fragment */
#define gry_KEYFRAGunits mp_bytes2units(gry_KEYFRAGSIZE)
#define gry_KEYFRAGbytes mp_units2bytes(gry_KEYFRAGunits)

LOCALPROC gry_extract_keyID(ui3p keyID, mp_UNIT_p n)
/*
	Extract key fragment from modulus n.  keyID byte array
	must be at least gry_KEYFRAGSIZE bytes long.
*/
{
	ui3b buf[gry_KEYFRAGbytes];
	ui4r save_precision = mp_precision;
		/* save mp_precision */

	mp_precision_set(gry_KEYFRAGunits);
	mp_convertoLSB(n, buf);
	mp_precision_set(save_precision);

	MyMoveBytes(buf, keyID, gry_KEYFRAGSIZE);
}

/* --- reading and writing multiprecision integers --- */

/*
	Read a multiprecision integer from a file, of
	mp_precision size. (mp_precision size should be a
	multiple of 128 bits, so as to work the same with any
	expected mp_UNIT_ln2bsz.)
*/
LOCALFUNC tMyErr gry_read_fixed_mpi(mp_UNIT_p r)
{
	tMyErr err;
	ui3b buf[MAX_BYTE_PRECISION];

	if (kMyErr_noErr == (err =
		ReadBuffToPtr(buf,
			mp_units2bytes(mp_precision))))
	{
		mp_converfromLSB(r, buf);
	}

	return ErrReportStack(err, "gry_read_fixed_mpi");
}

/*
	Write a multiprecision integer to a file, of
	mp_precision size. (mp_precision size should be a
	multiple of 128 bits, so as to work the same with any
	expected mp_UNIT_ln2bsz.)
*/
LOCALFUNC tMyErr gry_write_fixed_mpi(mp_UNIT_p n)
{
	tMyErr err;
	ui3b buf[MAX_BYTE_PRECISION];

	mp_convertoLSB(n, buf);

	err = WriteBuffFromPtr(buf,
		mp_units2bytes(mp_precision));

	return ErrReportStack(err, "gry_write_fixed_mpi");
}

LOCALFUNC tMyErr gry_read_mpi(mp_UNIT_p r)
{
	tMyErr err;
	uimr bitcount;
	ui4r bytecount;
	ui4r prec_bytes;
	ui3b buf[MAX_BYTE_PRECISION];

	if (kMyErr_noErr != (err =
		ReadArbIntNorm(&bitcount)))
	{
		goto l_exit;
	}

	bytecount = bits2bytes(bitcount);
	prec_bytes = mp_units2bytes(mp_precision);

	if (bytecount > prec_bytes) {
		err = kMyErrParamErr;
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		ReadBuffToPtr(buf, bytecount)))
	{
		goto l_exit;
	}

	fill0(&buf[bytecount], prec_bytes - bytecount);
		/* fill trailing zero bytes */

	mp_converfromLSB(r, buf);

	err = kMyErr_noErr;

l_exit:
	return ErrReportStack(err, "gry_read_mpi");
}

/*
	Write a multiprecision integer to a file.
*/
LOCALFUNC tMyErr gry_write_mpi(mp_UNIT_p n)
{
	tMyErr err;
	ui4r bitcount;
	ui4r bytecount;
	ui3b buf[MAX_BYTE_PRECISION];

	bitcount = mp_countbits(n);

	if (kMyErr_noErr != (err =
		WriteArbIntNorm_v2(bitcount)))
	{
		goto l_exit;
	}

	bytecount = bits2bytes(bitcount);
#if 0
	ui4r prec_bytes = mp_units2bytes(mp_precision);
	ui4r zero_bytes = prec_bytes - bytecount;
#endif

	mp_convertoLSB(n, buf);

	if (kMyErr_noErr != (err =
		WriteBuffFromPtr(buf, bytecount)))
	{
		/* fail */
		goto l_exit;
	}

	err = kMyErr_noErr;

l_exit:
	return ErrReportStack(err, "gry_write_mpi");
}


/* --- reading and writing GRY signatures and keys --- */

LOCALFUNC tMyErr ReadBuffVerifyBytes(ui3b *p, uimr L, ui3b *tmp)
{
	tMyErr err;

	if (kMyErr_noErr != (err = ReadBuffToPtr(tmp, L))) {
		goto l_exit;
	}

	if (! equal_buffers(tmp, p, L)) {
#if DebugCheck
		dbglog_writeln("differs");
		debug_dump_buffer("tmp", tmp, L);
		debug_dump_buffer("p", p, L);
#endif
		err = kMyErr_crptfl;
		goto l_exit;
	}

l_exit:
	return ErrReportStack(err, "ReadBuffVerifyBytes");
}


/*
	reading and writing GRY format signatures.
*/

LOCALVAR const ui3b gry_sig_idA[3] = {
	0x1A, 0xBC, 0xBF
};

LOCALVAR const ui3b gry_sig_idB[1] = {
	0xFC
};

/* Check signature in infile for validity. */
LOCALFUNC tMyErr gry_check_read_signature(
	mp_UNIT_p inbuf, ui3p keyID)
{
	tMyErr err;
	ui3b idA[sizeof(gry_sig_idA)];
	ui3b idB[sizeof(gry_sig_idB)];
	ui3b keyID0[gry_KEYFRAGSIZE];

	if (kMyErr_noErr != (err =
		ReadBuffVerifyBytes((ui3b *)gry_sig_idA, sizeof(gry_sig_idA),
			idA)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		ReadBuffToPtr(keyID0, gry_KEYFRAGSIZE)))
	{
		goto l_exit;
	}

	if (! equal_buffers(keyID, keyID0, gry_KEYFRAGSIZE)) {
#if DebugCheck
		dbglog_writeln(
"WARNING: Can't find the right public key-- can't check signature"
" integrity.");
#endif
		err = kMyErr_WrongPublicKey;
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		ReadBuffVerifyBytes((ui3b *)gry_sig_idB, sizeof(gry_sig_idB),
			idB)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err = gry_read_fixed_mpi(inbuf))) {
		goto l_exit;
	}

	if (! ReadBuffEOF()) {
#if DebugCheck
		dbglog_writeln("sig too long");
#endif
		err = kMyErr_crptfl;
		goto l_exit;
	}

	err = kMyErr_noErr;

l_exit:
	return ErrReportStack(err, "gry_check_read_signature");
}

/*
	Constructs a signed message digest in a signature certificate.
*/
LOCALFUNC tMyErr
gry_make_signature_certificate(
	mp_UNIT_p outbuf, ui3p keyID)
{
	tMyErr err;

	if (kMyErr_noErr != (err =
		WriteBuffFromPtr((ui3b *)gry_sig_idA, sizeof(gry_sig_idA))))
	{
		goto l_exit;
	}

	/* Now append keyID... */
	if (kMyErr_noErr != (err =
		WriteBuffFromPtr(keyID, gry_KEYFRAGSIZE)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		WriteBuffFromPtr((ui3b *)gry_sig_idB, sizeof(gry_sig_idB))))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		gry_write_fixed_mpi(outbuf)))
	{
		goto l_exit;
	}

	err = kMyErr_noErr;

l_exit:
	return err;
}

LOCALVAR const ui3b gry_pk_idA[5] = {
	0x1A, 0xBC, 0xBE, 0x3C, 0xA0
};

LOCALVAR const ui3b gry_pk_idB[3] = {
	0x00, 0x00, 0x3F
};

/*
	Padded so that with usual key size, ascii armored
	public key is exactly 3 lines long. Also, with one
	byte to represent size of n, n itself starts at
	byte 9, and so at character 12 of ascii armored key.
	And also, padding reserves room, in case need
	version field later.
*/

LOCALFUNC tMyErr gry_getpublickey_readbuff(mp_UNIT_p n, mp_UNIT_p e)
{
	tMyErr err;
	uimr bitcount;
	ui3b idA[sizeof(gry_pk_idA)];
	ui3b idB[sizeof(gry_pk_idB)];

	if (kMyErr_noErr != (err =
		ReadBuffVerifyBytes((ui3b *)gry_pk_idA, sizeof(gry_pk_idA),
			idA)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		ReadArbIntNorm(&bitcount)))
	{
		goto l_exit;
	}

	bitcount *= 128;

	if ((0 == bitcount) || (bitcount > MAX_BIT_PRECISION)) {
		err = kMyErrParamErr;
		goto l_exit;
	}

	mp_precision_set(mp_bits2units(bitcount));

	if (kMyErr_noErr != (err =
		ReadBuffVerifyBytes((ui3b *)gry_pk_idB, sizeof(gry_pk_idB),
			idB)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err = gry_read_fixed_mpi(n))) {
		goto l_exit;
	}

	/* Note that precision was adjusted for n */
	if (kMyErr_noErr != (err = gry_read_mpi(e))) {
		goto l_exit;
	}

	/*
		Note that readkeypacket has called mp_precision_set
	*/

	if (! ReadBuffEOF()) {
#if DebugCheck
		dbglog_writeln("key too long");
#endif
		err = -2;
		goto l_exit;
	}

	err = kMyErr_noErr;

l_exit:
	return ErrReportStack(err, "gry_getpublickey_readbuff");
}

LOCALFUNC tMyErr gry_writekeypacket_CERT_PUBKEY(
	mp_UNIT_p n, mp_UNIT_p e)
/*
	Write key components p, q, n, e, d, and u to specified file.
*/
{
	tMyErr err;

	if (kMyErr_noErr != (err =
		WriteBuffFromPtr((ui3b *)gry_pk_idA, sizeof(gry_pk_idA))))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		WriteArbIntNorm_v2(mp_units2bits(mp_precision) / 128)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		WriteBuffFromPtr((ui3b *)gry_pk_idB, sizeof(gry_pk_idB))))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		gry_write_fixed_mpi(n)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		gry_write_mpi(e)))
	{
		goto l_exit;
	}

l_exit:
	return ErrReportStack(err, "gry_writekeypacket_CERT_PUBKEY");
}

LOCALVAR const ui3b gry_sk_idA[5] = {
	0x1A, 0xBC, 0xBE, 0x48, 0xA0
};

LOCALVAR const ui3b gry_sk_idB[3] = {
	0x00, 0x00, 0x3F
};

/*
	make padding same size as for public key, so n will
	start in same place in ascii armored version.
*/

LOCALFUNC tMyErr gry_getsecretkey_readbuff(mp_UNIT_p n, mp_UNIT_p e,
	mp_UNIT_p d, mp_UNIT_p p, mp_UNIT_p q, mp_UNIT_p u)
{
	tMyErr err;
	uimr bitcount;
	ui3b idA[sizeof(gry_sk_idA)];
	ui3b idB[sizeof(gry_sk_idB)];

	if (kMyErr_noErr != (err =
		ReadBuffVerifyBytes((ui3b *)gry_sk_idA, sizeof(gry_sk_idA),
			idA)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		ReadArbIntNorm(&bitcount)))
	{
		goto l_exit;
	}

	bitcount *= 128;

	if ((0 == bitcount) || (bitcount > MAX_BIT_PRECISION)) {
		err = kMyErrParamErr;
		goto l_exit;
	}

	mp_precision_set(mp_bits2units(bitcount));

	if (kMyErr_noErr != (err =
		ReadBuffVerifyBytes((ui3b *)gry_sk_idB, sizeof(gry_sk_idB),
			idB)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err = gry_read_fixed_mpi(n))) {
		goto l_exit;
	}

	/* Note that precision was adjusted for n */
	if (kMyErr_noErr != (err = gry_read_mpi(e))) {
		goto l_exit;
	}

	if (kMyErr_noErr != (err = gry_read_mpi(d))) {
		goto l_exit;
	}
	if (kMyErr_noErr != (err = gry_read_mpi(p))) {
		goto l_exit;
	}
	if (kMyErr_noErr != (err = gry_read_mpi(q))) {
		goto l_exit;
	}

	/* use register 'u' briefly as scratchpad */
	mp_mult(u, p, q); /* compare p * q against n */
	if (! mp_eq(n, u)) { /* bad pass phrase? */
		err = -5; /* possible bad pass phrase, error return */
		goto l_exit;
	}

	/* now read in real u */
	if (kMyErr_noErr != (err = gry_read_mpi(u))) {
		goto l_exit;
	}

	if (mp_isZero(d)) {
		/* didn't get secret key components */
#if DebugCheck
		dbglog_writeln(
			"Key file is not a secret key file.");
#endif
		err = -2;
		goto l_exit;
	}

	if (! ReadBuffEOF()) {
#if DebugCheck
		dbglog_writeln("key too long");
#endif
		err = -2;
		goto l_exit;
	}

	err = kMyErr_noErr;

l_exit:
	return ErrReportStack(err, "gry_getsecretkey_readbuff");
}

LOCALFUNC tMyErr grc_writekeypacket_CERT_SECKEY(
	mp_UNIT_p n, mp_UNIT_p e, mp_UNIT_p d, mp_UNIT_p p, mp_UNIT_p q,
	mp_UNIT_p u)
{
	tMyErr err;

	if (kMyErr_noErr != (err =
		WriteBuffFromPtr((ui3b *)gry_sk_idA, sizeof(gry_sk_idA))))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		WriteArbIntNorm_v2(mp_units2bits(mp_precision) / 128)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		WriteBuffFromPtr((ui3b *)gry_sk_idB, sizeof(gry_sk_idB))))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		gry_write_fixed_mpi(n)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		gry_write_mpi(e)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		gry_write_mpi(d)))
	{
		goto l_exit;
	}
	if (kMyErr_noErr != (err =
		gry_write_mpi(p)))
	{
		goto l_exit;
	}
	if (kMyErr_noErr != (err =
		gry_write_mpi(q)))
	{
		goto l_exit;
	}
	if (kMyErr_noErr != (err =
		gry_write_mpi(u)))
	{
		goto l_exit;
	}

l_exit:
	return ErrReportStack(err, "pgp_writekeypacket_CERT_SECKEY");
}

LOCALFUNC tMyErr gry_check_read_signature_H(MyHandle inH,
	mp_UNIT_p inbuf, ui3p keyID)
{
	tMyErr err;

	if (kMyErr_noErr != (err =
		ReadFromBuffXhndBegin(inH)))
	{
		/* fail */
	} else
	{
		err = gry_check_read_signature(
			inbuf, keyID);

		err = ReadBuffFromXhndEnd(err);
	}

	return ErrReportStack(err, "gry_check_read_signature_H");
}

LOCALFUNC tMyErr
gry_make_signature_certificate_H(
	mp_UNIT_p outbuf, ui3p keyID,
	MyHandle *outH)
{
	tMyErr err;

	if (kMyErr_noErr != (err =
		WriteBuffXhndBegin()))
	{
		/* fail */
	} else {
		err = gry_make_signature_certificate(
				outbuf, keyID);

		err = WriteBuffXhndEnd(err, outH);
	}

	return ErrReportStack(err, "gry_make_signature_certificate_H");
}

LOCALFUNC tMyErr gry_getpublickey(MyHandle pubring_H,
	mp_UNIT_p n, mp_UNIT_p e)
{
	tMyErr err;

	if (kMyErr_noErr != (err =
		ReadFromBuffXhndBegin(pubring_H)))
	{
		/* fail */
	} else {
		err = gry_getpublickey_readbuff(n, e);

		err = ReadBuffFromXhndEnd(err);
	}

	return ErrReportStack(err, "gry_getpublickey");
}

LOCALFUNC tMyErr gry_writekeyfile_CERT_PUBKEY(MyHandle *pubkey_H,
	mp_UNIT_p n, mp_UNIT_p e)
/*
	Write key components p, q, n, e, d, and u to specified file.
	hidekey is TRUE iff key should be encrypted.
	userid is a length-prefixed Pascal-type character string.
	We write three packets: a key packet, a key control packet, and
	a userid packet.  We assume the key being written is our own,
	so we set the control bits for full trust.
*/
{
	tMyErr err;

	if (kMyErr_noErr != (err =
		WriteBuffXhndBegin()))
	{
		/* fail */
	} else {
		err = gry_writekeypacket_CERT_PUBKEY(n, e);

		err = WriteBuffXhndEnd(err, pubkey_H);
	}

	return err;
}

LOCALFUNC tMyErr gry_getsecretkey_H(MyHandle secring_H,
	mp_UNIT_p n, mp_UNIT_p e,
	mp_UNIT_p d, mp_UNIT_p p, mp_UNIT_p q, mp_UNIT_p u)
{
	tMyErr err;

	if (kMyErr_noErr != (err =
		ReadFromBuffXhndBegin(secring_H)))
	{
		/* fail */
	} else
	{
		err = gry_getsecretkey_readbuff(n, e, d, p, q, u);

		err = ReadBuffFromXhndEnd(err);
	}

	return ErrReportStack(err, "gry_getsecretkey_H");
}

LOCALFUNC tMyErr gry_writekeyfile_CERT_SECKEY(MyHandle *seckey_H,
	mp_UNIT_p n, mp_UNIT_p e, mp_UNIT_p d, mp_UNIT_p p, mp_UNIT_p q,
	mp_UNIT_p u)
/*
	Write key components p, q, n, e, d, and u to specified file.
*/
{
	tMyErr err;

	if (kMyErr_noErr != (err =
		WriteBuffXhndBegin()))
	{
		/* fail */
	} else {
		err = grc_writekeypacket_CERT_SECKEY(n, e, d, p, q, u);

		err = WriteBuffXhndEnd(err, seckey_H);
	}

	return err;
}


/*
	encrypting and decrypting for GRY format signatures.
	simpler than the PGP style
*/

LOCALFUNC tMyErr
gry_public_decrypt(ui3p outbuf, mp_UNIT_p inbuf,
	mp_UNIT_p E, mp_UNIT_p N, ui4r bytes)
/*
	Decrypt a message digest using a public key.
*/
{
	tMyErr err;
	mp_UNIT_t temp0[MAX_UNIT_PRECISION];
	uimr nUnits = bytes >> mp_UNIT_ln2sz;

	if (bytes != nUnits << mp_UNIT_ln2sz) {
#if DebugCheck
		dbglog_writeln("bytes not even multiple");
#endif
		err = -9;
		goto l_exit;
	}

	if (nUnits >= mp_significance(N)) {
#if DebugCheck
		dbglog_writeln("nUnits too big");
#endif
		err = -9;
		goto l_exit;
	}

	if (kMyErr_noErr != (err = mp_modexp(temp0, inbuf, E, N))) {
		goto l_exit;
	}

	if (! mp_RestIsZero(temp0, nUnits)) {
#if DebugCheck
		dbglog_writeln("! mp_RestIsZero");
#endif
		err = -9;
		goto l_exit;
	}

	{
		ui4r save_precision = mp_precision;

		mp_precision_set(nUnits);

		mp_convertoLSB(temp0, outbuf);

		mp_precision_set(save_precision);
	}

	err = kMyErr_noErr;

l_exit:
	return ErrReportStack(err, "gry_public_decrypt");
}

LOCALFUNC tMyErr
gry_private_encrypt(mp_UNIT_p outbuf, ui3p inbuf, ui4r bytes,
	mp_UNIT_p E, mp_UNIT_p D, mp_UNIT_p P, mp_UNIT_p Q,
	mp_UNIT_p U, mp_UNIT_p N)
/*
	Encrypt a message digest with a private key.
*/
{
	tMyErr err;
	mp_UNIT_t temp0[MAX_UNIT_PRECISION];
	uimr nUnits = bytes >> mp_UNIT_ln2sz;

	if (bytes != nUnits << mp_UNIT_ln2sz) {
		err = -9;
		goto l_exit;
	}

	if (nUnits >= mp_significance(N)) {
		err = -9;
		goto l_exit;
	}

	{
		ui4r save_precision = mp_precision;

		mp_precision_set(nUnits);

		mp_converfromLSB(temp0, inbuf);

		mp_precision_set(save_precision);

		mp_SetRestZero(temp0, nUnits);
	}

	(void)E; /* unused */

#if 0
	err = mp_modexp(outbuf, temp0, D, N);
#else
	/* This gets same result, but much faster */

	/*
		These coefficents aren't stored in private key, so
		compute them here.
	*/
	{
		mp_UNIT_t t[MAX_UNIT_PRECISION];
		mp_UNIT_t DP[MAX_UNIT_PRECISION];
		mp_UNIT_t DQ[MAX_UNIT_PRECISION];

		mp_move(t, P);
		mp_dec(t);
		mp_mod(DP, D, t);
		mp_move(t, Q);
		mp_dec(t);
		mp_mod(DQ, D, t);

		err = mp_modexp_crt(outbuf, temp0, P, Q, DP, DQ, U);
	}
#endif

l_exit:
	return ErrReportStack(err, "gry_private_encrypt");
}

#define gry_CRCSIZE 3

LOCALFUNC tMyErr CrcSaveLSBFromJustHandle(ui3p p,
	MyHandle h)
{
	tMyErr err;
	crcword crc;

	if (kMyErr_noErr == (err =
		CrcFromJustHandle(h, &crc)))
	{
		p[0] = (crc      ) & 0xff;
		p[1] = (crc >>  8) & 0xff;
		p[2] = (crc >> 16) & 0xff;
	}

	return ErrReportStack(err, "CrcSaveLSBFromJustHandle");
}

LOCALPROC MyBytesReverse(ui3p p, uimr L)
{
	uimr i;
	ui3b t;
	ui3p pA = p;
	ui3p pB = p + L;

	for (i = (L / 2) + 1; 0 != --i; ) {
		t = *pA;
		*pA++ = *--pB;
		*pB = t;
	}
}

LOCALFUNC tMyErr MyHandReverse(MyHandle h)
{
	tMyErr err;
	uimr L;

	if (kMyErr_noErr == (err =
		MyHandleGetSize_v2(h, &L)))
	{
		MyHandleLock(h);
		MyBytesReverse((ui3p)MyHandleP(h), L);
		MyHandleUnlock(h);
	}

	return ErrReportStack(err, "MyHandReverse");
}

#define gry_dgof_md5 2 /* 2 */
#define gry_dgof_rmd5 18 /* 2 + 16 */
#define gry_dgof_crc 34 /* 2 + 16 + 16 */
#define gry_dgof_rcrc 37 /* 2 + 16 + 16 + 3 */

#define gry_digest_sz 40 /* 2 + 16 + 16 + 3 + 3 */

LOCALFUNC tMyErr gry_calc_digest(ui3p digest,
	MyHandle inH)
{
	tMyErr err;

	/*
		assuming inH already in canonical form
		(read_gry_signedfile_message)
	*/

	/* version info can go in first two bytes */
	digest[0] = 1;
	digest[1] = 0;

	if (kMyErr_noErr != (err =
		MDfileFromJustHandle(digest + gry_dgof_md5, inH)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		CrcSaveLSBFromJustHandle(digest + gry_dgof_crc, inH)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		MyHandReverse(inH)))
	{
		goto l_exit;
	}


	if (kMyErr_noErr != (err =
		MDfileFromJustHandle(digest + gry_dgof_rmd5, inH)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		CrcSaveLSBFromJustHandle(digest + gry_dgof_rcrc, inH)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		MyHandReverse(inH)))
	{
		goto l_exit;
	}

	err = kMyErr_noErr;

l_exit:
	return ErrReportStack(err, "gry_calc_digest");
}

LOCALFUNC tMyErr gry_check_signaturefile(MyHandle inH, MyHandle litH,
	MyHandle PubringH)
{
	tMyErr err;
	mp_UNIT_t n[MAX_UNIT_PRECISION];
	mp_UNIT_t e[MAX_UNIT_PRECISION];
	ui3b keyID[gry_KEYFRAGSIZE];
	mp_UNIT_t inbuf[MAX_UNIT_PRECISION];
	ui3b digest[gry_digest_sz];
	ui3b outbuf[MAX_BYTE_PRECISION];

	if (kMyErr_noErr != (err =
		gry_getpublickey(PubringH, n, e)))
	{
		goto l_exit;
	}

	gry_extract_keyID(keyID, n);

	if (kMyErr_noErr != (err =
		gry_check_read_signature_H(inH,
			inbuf, keyID)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		gry_calc_digest(digest, litH)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		gry_public_decrypt(outbuf, inbuf, e, n, gry_digest_sz)))
	{
		goto l_exit;
	}

	/* now compare computed MD with claimed MD */
	if (! equal_buffers(digest, outbuf, gry_digest_sz)) {
#if DebugCheck
		dbglog_writeln(
"WARNING: Bad signature, doesn't match file contents!");
#endif
		err = kMyErr_BadSignature;
		goto l_exit;
	}

	err = kMyErr_noErr;

l_exit:
	return ErrReportStack(err, "gry_check_signaturefile");
}

LOCALFUNC tMyErr gry_signfile(MyHandle inH,
	MyHandle secring_H, MyHandle *outH)
{
	tMyErr err;
	mp_UNIT_t n[MAX_UNIT_PRECISION], e[MAX_UNIT_PRECISION];
	mp_UNIT_t d[MAX_UNIT_PRECISION];
	mp_UNIT_t p[MAX_UNIT_PRECISION], q[MAX_UNIT_PRECISION];
	mp_UNIT_t u[MAX_UNIT_PRECISION];
	ui3b keyID[gry_KEYFRAGSIZE];
	ui3b digest[gry_digest_sz];
	mp_UNIT_t outbuf[MAX_UNIT_PRECISION];

	if (kMyErr_noErr == (err =
		gry_getsecretkey_H(secring_H, n, e, d, p, q, u)))
	{
		gry_extract_keyID(keyID, n); /* gets keyID */

		if (mp_countbits(n) < 256) {
#if DebugCheck
			dbglog_writeln(
				"Error: RSA key length must be at least 256 bits.");
#endif
			err = kMyErrParamErr;
		} else
		if (kMyErr_noErr != (err =
			gry_calc_digest(digest, inH)))
		{
			/* fail */
		} else
		if (kMyErr_noErr != (err =
			gry_private_encrypt(outbuf, digest, gry_digest_sz,
				e, d, p, q, u, n)))
		{
			/* fail */
		} else
		{
			err = gry_make_signature_certificate_H(
				outbuf, keyID, outH);
		}
	}

	return ErrReportStack(err, "gry_signfile");
}
#ifdef Have_PKGNKEYS
LOCALFUNC tMyErr gry_dokeygen_0(MyHandle random_H,
	MyHandle *seckey_H, MyHandle *pubkey_H)
{
	tMyErr err;
	mp_UNIT_t n[MAX_UNIT_PRECISION], e[MAX_UNIT_PRECISION];
	mp_UNIT_t d[MAX_UNIT_PRECISION], p[MAX_UNIT_PRECISION];
	mp_UNIT_t q[MAX_UNIT_PRECISION], u[MAX_UNIT_PRECISION];

	if (kMyErr_noErr != (err =
		rsa_keygen1(random_H, n, e, d, p, q, u)))
	{
		/* fail */
	} else
	{
		if (kMyErr_noErr == (err =
			gry_writekeyfile_CERT_SECKEY(seckey_H, n, e, d, p, q, u)))
		{
			if (kMyErr_noErr == (err =
				gry_writekeyfile_CERT_PUBKEY(pubkey_H, n, e)))
			{
				err = kMyErr_noErr;

#if 0
				if (kMyErr_noErr != err) {
					(void) MyHandleDispose_v2(*pubkey_H);
					*pubkey_H = nullpr;
				}
#endif
			}
			if (kMyErr_noErr != err) {
				(void) MyHandleDispose_v2(*seckey_H);
				*seckey_H = nullpr;
			}
		}
	}

	return ErrReportStack(err, "gry_dokeygen_0");
}
#endif

LOCALFUNC tMyErr MyHandAppendCRC(MyHandle h)
{
	tMyErr err;
	uimr L;
	ui3b crc[gry_CRCSIZE];

	if (kMyErr_noErr != (err =
		MyHandleGetSize_v2(h, &L)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		CrcSaveLSBFromJustHandle(crc, h)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		MyHandleSetSize_v2(h, L + gry_CRCSIZE)))
	{
		goto l_exit;
	}

	MyHandleLock(h);
	MyMoveBytes(crc, (ui3p)MyHandleP(h) + L, gry_CRCSIZE);
	MyHandleUnlock(h);

l_exit:
	return ErrReportStack(err, "MyHandAppendCRC");
}

LOCALFUNC tMyErr MyHandChkRmvCRC(MyHandle h)
{
	tMyErr err;
	uimr L;
	ui3b crc0[gry_CRCSIZE];
	ui3b crc[gry_CRCSIZE];

	if (kMyErr_noErr != (err =
		MyHandleGetSize_v2(h, &L)))
	{
		goto l_exit;
	}

	MyHandleLock(h);
	MyMoveBytes((ui3p)MyHandleP(h) + L - gry_CRCSIZE,
		crc0, gry_CRCSIZE);
	MyHandleUnlock(h);

	if (kMyErr_noErr != (err =
		MyHandleSetSize_v2(h, L - gry_CRCSIZE)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		CrcSaveLSBFromJustHandle(crc, h)))
	{
		goto l_exit;
	}

	if (! equal_buffers(crc0, crc, gry_CRCSIZE)) {
#if DebugCheck
		dbglog_writeln(
"WARNING: Bad crc checksum!");
#endif
#if 1
		err = kMyErr_crptfl;
		goto l_exit;
#endif
	}

	err = kMyErr_noErr;

l_exit:
	return ErrReportStack(err, "MyHandChkRmvCRC");
}

LOCALFUNC tMyErr ArmorFileDecodeChkRmvCRC(MyHandle inH,
	uimr offsetA, uimr offsetB,
	MyHandle *outH)
{
	tMyErr err;

	if (kMyErr_noErr != (err =
		ArmorFileDecode_H(inH, offsetA, offsetB, outH)))
	{
		if (kMyErr_UndesiredChar == err) {
			err = SetSyntaxErrCStr("armor not valid",
				offsetA, offsetB);
		}
	} else
	{
		err = MyHandChkRmvCRC(*outH);
	}

	return ErrReportStack(err, "ArmorFileDecodeChkRmvCRC");
}

LOCALFUNC tMyErr
write_armoredAppendCRC(MyHandle inH)
{
	tMyErr err;
	uimr L;

	if (kMyErr_noErr == (err =
		MyHandleGetSize_v2(inH, &L)))
	{
		if (kMyErr_noErr == (err =
			MyHandAppendCRC(inH)))
		{
			err = write_armored_H(inH);
		}

		err = ErrCombine(err,
			MyHandleSetSize_v2(inH, L));
	}

	return ErrReportStack(err, "write_armoredAppendCRC");
}

LOCALVAR const char begin_gry_signed_msg[]
	= "--------- GRY SIGNED TEXT ---------";

LOCALVAR const char begin_gry_sig[]
	= "------- BEGIN GRY SIGNATURE -------";

LOCALVAR const char end_gry_sig[]
	= "-------- END GRY SIGNATURE --------";

LOCALVAR const char begin_gry_public_key[]
	= "----- BEGIN GRY PUBLIC KEY -----";

LOCALVAR const char end_gry_public_key[]
	= "------ END GRY PUBLIC KEY ------";

LOCALVAR const char begin_gry_secret_key[]
	= "----- BEGIN GRY SECRET KEY -----";

LOCALVAR const char end_gry_secret_key[]
	= "------ END GRY SECRET KEY ------";

LOCALFUNC tMyErr gry_de_armor_signedfile0(MyHandle inH,
	uimr o1, uimr o2,
	MyHandle *cipherH, MyHandle *litH)
{
	tMyErr err;
	uimr o9;
	uimr o10;
	uimr o_ta;
	uimr o_tb;
	MyHandle litH0 = nullpr;
	MyHandle cipherH0 = nullpr;

	if (kMyErr_noErr != (err =
		MyHandRngFindLastLnRmvSpc(inH, o1, o2, &o_ta, &o_tb)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		MyHandRngEqualCStr(inH, o_tb, o2,
			(char *)end_gry_sig)))
	{
		if (kMyErrNoMatch == err) {
			err = SetSyntaxErrCStr(
				"This is not the correct end of a GRY signature",
				o_tb, o2);
		}
		goto l_exit;
	}

	o2 = o_ta;


	if (kMyErr_noErr != (err =
		MyHandRngFindLastMatchLine(inH, o1, o2,
			(ui3p)begin_gry_sig,
			sizeof(begin_gry_sig) - 1,
			&o9, &o10)))
	{
		if (kMyErrNoMatch == err) {
			err = SetSyntaxErrCStr("There is no begin signature line",
				o1, o2);
		}
		goto l_exit;
	}


	if (kMyErr_noErr != (err =
		MyHandRngFindLastLnRmvSpc(inH, o1, o9, &o_ta, &o_tb)))
	{
		goto l_exit;
	}

	if (o_tb != o9) {
		err = SetSyntaxErrCStr(
			"A blank line is expected after the message body",
			o_tb, o9);
		goto l_exit;
	}

	o9 = o_ta;


	if (kMyErr_noErr != (err =
		MyHandRngFindFirstLnRmvSpc(inH, o1, o9, &o_tb, &o_ta)))
	{
		goto l_exit;
	}

	if (o1 != o_tb) {
		err = SetSyntaxErrCStr(
			"A blank line is expected before the message body",
			o1, o_tb);
		goto l_exit;
	}

	o1 = o_ta;


	if (kMyErr_noErr != (err =
		read_gry_signedfile_message_H(inH, o1, o9, &litH0)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		ArmorFileDecodeChkRmvCRC(inH, o10, o2, &cipherH0)))
	{
		goto l_exit;
	}

	err = kMyErr_noErr;

l_exit:

	if (kMyErr_noErr == err) {
		*cipherH = cipherH0;
		*litH = litH0;
	} else {
		if (nullpr != cipherH0) {
			(void) MyHandleDispose_v2(cipherH0);
		}
		if (nullpr != litH0) {
			(void) MyHandleDispose_v2(litH0);
		}
	}

	return ErrReportStack(err, "gry_de_armor_signedfile0");
}

LOCALFUNC tMyErr gry_de_armor_signedfile(MyHandle inputH,
	MyHandle *cipherH, MyHandle *litH)
{
	tMyErr err;
	uimr o1;
	uimr o2;
	uimr o_ta;
	uimr o_tb;

	if (kMyErr_noErr != (err =
		MyHandFindLeadingTrailingSpace(inputH, &o1, &o2)))
	{
		goto l_exit;
	}


	if (kMyErr_noErr != (err =
		MyHandRngFindFirstLnRmvSpc(inputH, o1, o2, &o_tb, &o_ta)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		MyHandRngEqualCStr(inputH, o1, o_tb,
			(char *)begin_gry_signed_msg)))
	{
		if (kMyErrNoMatch == err) {
			err = SetSyntaxErrCStr(
				"This does not look like the start of a signed message",
				o1, o_tb);
		}
		goto l_exit;
	}

	o1 = o_ta;


	err = gry_de_armor_signedfile0(inputH, o1, o2,
		cipherH, litH);

l_exit:
	return ErrReportStack(err, "gry_de_armor_signedfile");
}

LOCALFUNC tMyErr gry_de_armor_keyfile(MyHandle keyH, MyHandle *outH,
	blnr is_secret_key)
{
	tMyErr err;
	uimr o1;
	uimr o2;
	uimr o_ta;
	uimr o_tb;

	if (kMyErr_noErr != (err =
		MyHandFindLeadingTrailingSpace(keyH, &o1, &o2)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		MyHandRngFindFirstLnRmvSpc(keyH, o1, o2, &o_tb, &o_ta)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		MyHandRngEqualCStr(keyH, o1, o_tb,
			is_secret_key
				? (char *)begin_gry_secret_key
				: (char *)begin_gry_public_key)))
	{
		if (kMyErrNoMatch == err) {
			err = SetSyntaxErrCStr(
				"This doesn't look like the start of a key",
				o1, o2);
		}
		goto l_exit;
	}

	o1 = o_ta;


	if (kMyErr_noErr != (err =
		MyHandRngFindLastLnRmvSpc(keyH, o1, o2, &o_ta, &o_tb)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		MyHandRngEqualCStr(keyH, o_tb, o2,
			is_secret_key
				? (char *)end_gry_secret_key
				: (char *)end_gry_public_key)))
	{
		if (kMyErrNoMatch == err) {
			err = SetSyntaxErrCStr(
				"This doesn't look like the end of a key",
				o1, o2);
		}
		goto l_exit;
	}

	o2 = o_ta;


	if (kMyErr_noErr != (err =
		ArmorFileDecodeChkRmvCRC(keyH, o1, o2, outH)))
	{
		goto l_exit;
	}


	err = kMyErr_noErr;

l_exit:
	return ErrReportStack(err, "gry_de_armor_keyfile");
}

LOCALFUNC tMyErr
gry_armor_signfile(MyHandle inH, MyHandle tempH)
{
	tMyErr err;

	if (kMyErr_noErr != (err =
		WriteBuffFromCStrAndCR((char *)begin_gry_signed_msg)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		WriteBuffCharCR()))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		WriteBuffFromHand(inH)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		WriteBuffCharCR()))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		WriteBuffCharCR()))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		WriteBuffFromCStrAndCR((char *)begin_gry_sig)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		write_armoredAppendCRC(tempH)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		WriteBuffFromCStrAndCR((char *)end_gry_sig)))
	{
		goto l_exit;
	}

	err = kMyErr_noErr;

l_exit:
	return ErrReportStack(err, "gry_armor_signfile");
}

LOCALFUNC tMyErr
gry_armor_keyfile(MyHandle inH, blnr is_secret_key)
{
	tMyErr err;

	if (kMyErr_noErr != (err =
		WriteBuffFromCStrAndCR(is_secret_key
			? (char *)begin_gry_secret_key
			: (char *)begin_gry_public_key)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		write_armoredAppendCRC(inH)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		WriteBuffFromCStrAndCR(is_secret_key
			? (char *)end_gry_secret_key
			: (char *)end_gry_public_key)))
	{
		goto l_exit;
	}

	err = kMyErr_noErr;

l_exit:
	return ErrReportStack(err, "gry_armor_keyfile");
}

LOCALFUNC tMyErr
gry_armor_signfile_H(MyHandle inH, MyHandle tempH, MyHandle *outH)
{
	tMyErr err;

	if (kMyErr_noErr != (err =
		WriteBuffXhndBegin()))
	{
		/* fail */
	} else {
		err = gry_armor_signfile(inH, tempH);

		err = WriteBuffXhndEnd(err, outH);
	}

	return ErrReportStack(err, "gry_armor_signfile_H");
}

LOCALFUNC tMyErr
gry_armor_keyfile_H(MyHandle inH, MyHandle *outH, blnr is_secret_key)
{
	tMyErr err;

	if (kMyErr_noErr != (err =
		WriteBuffXhndBegin()))
	{
		/* fail */
	} else {
		err = gry_armor_keyfile(inH, is_secret_key);

		err = WriteBuffXhndEnd(err, outH);
	}

	return ErrReportStack(err, "gry_armor_keyfile_H");
}

LOCALFUNC tMyErr gry_dispatch_checksig_0(MyHandle inputH,
	MyHandle keyH)
{
	tMyErr err;
	MyHandle PubringH;
	MyHandle cipherH;
	MyHandle litH;

	if (kMyErr_noErr == (err =
		gry_de_armor_signedfile(inputH,
			&cipherH, &litH)))
	{
		if (kMyErr_noErr == (err =
			gry_de_armor_keyfile(keyH, &PubringH, falseblnr)))
		{
			err = gry_check_signaturefile(cipherH, litH, PubringH);
			(void) MyHandleDispose_v2(PubringH);
		}
		(void) MyHandleDispose_v2(litH);
		(void) MyHandleDispose_v2(cipherH);
	}

	return ErrReportStack(err, "gry_dispatch_checksig_0");
}

LOCALFUNC tMyErr gry_dispatch_sign_1(MyHandle inH,
	MyHandle secring_H, MyHandle *outH)
{
	tMyErr err;
	uimr o1;
	uimr o2;
	MyHandle litH;
	MyHandle tempH;

	if (kMyErr_noErr != (err =
		MyHandFindLeadingTrailingSpace(inH, &o1, &o2)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr == (err =
		read_gry_signedfile_message_H(inH, o1, o2, &litH)))
	{
		if (kMyErr_noErr == (err =
			gry_signfile(litH, secring_H, &tempH)))
		{
			err = gry_armor_signfile_H(litH, tempH, outH);
			MyHandleDispose_v2(tempH);
		}
		MyHandleDispose_v2(litH);
	}

l_exit:
	return ErrReportStack(err, "gry_dispatch_sign_1");
}

#ifdef Have_PKGNKEYS
/*
	Do an RSA key pair generation, and write them out to the
	keyring files. numstr is a decimal string, the desired
	bitcount for the modulus n. numstr2 is a decimal string,
	the desired bitcount for the exponent e. username is the
	desired name for the key.
*/
LOCALFUNC int gry_dokeygen(MyHandle random_H,
	MyHandle *seckey_H, MyHandle *pubkey_H)
{
	tMyErr err;
	MyHandle randbin_H;
	MyHandle seckeybin_H;
	MyHandle pubkeybin_H;

	if (kMyErr_noErr == (err =
		ConvertHex2Bin_H(random_H, &randbin_H)))
	{
		if (kMyErr_noErr == (err =
			gry_dokeygen_0(randbin_H, &seckeybin_H, &pubkeybin_H)))
		{
			if (kMyErr_noErr == (err =
				gry_armor_keyfile_H(seckeybin_H, seckey_H, trueblnr)))
			{
				err = gry_armor_keyfile_H(pubkeybin_H, pubkey_H,
					falseblnr);

				if (kMyErr_noErr != err) {
					MyHandleDispose_v2(*seckey_H);
					*seckey_H = nullpr;
				}
			}

			MyHandleDispose_v2(seckeybin_H);
			MyHandleDispose_v2(pubkeybin_H);
		}
		MyHandleDispose_v2(randbin_H);
	}

	return err;
}
#endif
