/*
	RDWRSTRM.h
	Copyright (C) 2018 Paul C. Pratt

	You can redistribute this file and/or modify it under the terms
	of version 2 of the GNU General Public License as published by
	the Free Software Foundation.  You should have received a copy
	of the license along with this file; see the file COPYING.

	This file is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	license for more details.
*/

/*
	ReaD and WRite for STReaMs
*/

#ifndef UseStdFile
#define UseStdFile 0
#endif

enum {
	kReadBuffSrcNone,
#if UseStdFile
	kReadBuffSrcFile,
#endif
	kReadBuffSrcXhnd, /* existing handle */

	CountReadBuffDest
};

LOCALVAR ui3r ReadBuffSrc = kReadBuffSrcNone;
LOCALVAR uimr ReadBuffOffSet;
LOCALVAR uimr ReadBuffLength;


#if UseStdFile

LOCALVAR FILE *ReadBuffFP = nullpr;

LOCALFUNC tMyErr ReadBuffFromOpenFileBegin(FILE *f)
{
	tMyErr err;
	long fpos;
	long fpos2;

	if (kReadBuffSrcNone != ReadBuffSrc) {
		err = kMyErrProgramBug;
	} else
	if (nullpr != ReadBuffFP) {
		err = kMyErrProgramBug;
	} else
	if ((fpos = ftell(f)) < 0) {
		err = kMyErrSysUnknown;
	} else
	if (0 != fseek(f, 0L, SEEK_END)) {
		err = kMyErrSysUnknown;
	} else
	if ((fpos2 = ftell(f)) < 0) {
		err = kMyErrSysUnknown;
	} else
	if (0 != fseek(f, fpos, SEEK_SET)) {
		err = kMyErrSysUnknown;
	} else
	{
		err = kMyErr_noErr;

		ReadBuffSrc = kReadBuffSrcFile;
		ReadBuffFP = f;
		ReadBuffOffSet = fpos;
		ReadBuffLength = fpos2;
	}

	return ErrReportStack(err, "ReadBuffFromOpenFileBegin");
}

LOCALFUNC tMyErr ReadBuffFromOpenFileEnd(void)
{
	tMyErr err;

	if (kReadBuffSrcFile != ReadBuffSrc) {
		err = kMyErrProgramBug;
	} else
	if (nullpr == ReadBuffFP) {
		err = kMyErrProgramBug;
	} else
	{
		err = kMyErr_noErr;
	}

	ReadBuffFP = nullpr;
	ReadBuffSrc = kReadBuffSrcNone;

	return ErrReportStack(err, "ReadBuffFromOpenFileEnd");
}

LOCALFUNC tMyErr ReadBuffFromFilePathBegin(char *srcFile)
{
	tMyErr err;
	FILE *f;

	if ((f = fopen(srcFile, "rb")) == nullpr) {
		/* Can't open files */
		err = kMyErrSysUnknown;
	} else
	if (kMyErr_noErr != (err =
		ReadBuffFromOpenFileBegin(f)))
	{
		/* fail */
	} else
	{
		err = kMyErr_noErr;
	}

	return ErrReportStack(err, "ReadBuffFromFilePathBegin");
}

LOCALFUNC tMyErr ReadBuffFromFilePathEnd(tMyErr err0)
{
	tMyErr err;
	FILE *f = ReadBuffFP;

	if (nullpr == f) {
		err = kMyErr_noErr;
	} else {
		if (kMyErr_noErr != (err = ReadBuffFromOpenFileEnd())) {
			/* fail */
		}

		fclose(f);
	}

#if DebugCheck
	if (kMyErr_noErr != err) {
		err = ErrReportStack0(err, "ReadBuffFromFilePathEnd");
	}
#endif

	if (kMyErr_noErr != err0) {
		err = err0;
	}

	return err;
}

LOCALFUNC tMyErr ReadBuffFileGetOffset(uimr *i)
{
	*i = ReadBuffOffSet;

	return kMyErr_noErr;
#if 0
	tMyErr err;

	if ((*i = ftell(ReadBuffFP)) < 0) {
		err = kMyErrSysUnknown;
	} else
	{
		err = kMyErr_noErr;
	}

	return ErrReportStack(err, "ReadBuffFileGetOffset");
#endif
}

LOCALFUNC tMyErr ReadBuffFileRemaining(uimr *v)
{
	*v = ReadBuffLength - ReadBuffOffSet;

	return kMyErr_noErr;
#if 0
	tMyErr err;
	long fpos;
	long fpos2;

	if ((fpos = ftell(ReadBuffFP)) < 0) {
		err = kMyErrSysUnknown;
	} else
	if (0 != fseek(ReadBuffFP, 0L, SEEK_END)) {
		err = kMyErrSysUnknown;
	} else
	if ((fpos2 = ftell(ReadBuffFP)) < 0) {
		err = kMyErrSysUnknown;
	} else
	if (0 != fseek(ReadBuffFP, fpos, SEEK_SET)) {
		err = kMyErrSysUnknown;
	} else
	{
		err = kMyErr_noErr;
		*v = fpos2 - fpos;
	}

	return ErrReportStack(err, "ReadBuffFileRemaining");
#endif
}

LOCALFUNC tMyErr ReadBuffFileToPtr(MyPtr p, uimr L)
{
	tMyErr err;
	long n;

	n = fread(p, 1, L, ReadBuffFP);
	if (n < L) {
		err = kMyErrSysUnknown;
			/* error -- read failure or premature eof */
	} else {
		err = kMyErr_noErr;
		if (n > 0) {
			ReadBuffOffSet += L;
		}
	}

	return ErrReportStack(err, "ReadBuffFileToPtr");
}

LOCALFUNC blnr ReadBuffFileEOF(void)
{
	return (ReadBuffOffSet >= ReadBuffLength);

#if 0 /* does not seem to work in MPW */
	return feof(ReadBuffFP);
#endif
}

#endif /* UseStdFile */


LOCALVAR MyHandle ReadBuffHandle = nullpr;

LOCALFUNC tMyErr ReadFromBuffXhndRngBegin(MyHandle h,
	uimr offsetA, uimr offsetB)
{
	tMyErr err;

	if (kReadBuffSrcNone != ReadBuffSrc) {
		err = kMyErrProgramBug;
	} else
	if (nullpr != ReadBuffHandle) {
		err = kMyErrProgramBug;
	} else
	{
		err = kMyErr_noErr;

		ReadBuffSrc = kReadBuffSrcXhnd;
		ReadBuffHandle = h;
		ReadBuffOffSet = offsetA;
		ReadBuffLength = offsetB;
	}

	return ErrReportStack(err, "ReadFromBuffXhndRngBegin");
}

LOCALFUNC tMyErr ReadFromBuffXhndBegin(MyHandle h)
{
	tMyErr err;
	uimr L;

	if (kMyErr_noErr != (err =
		MyHandleGetSize_v2(h, &L)))
	{
		/* fail */
	} else {
		err = ReadFromBuffXhndRngBegin(h, 0, L);
	}

	return ErrReportStack(err, "ReadFromBuffXhndBegin");
}

LOCALFUNC tMyErr ReadBuffFromXhndEnd(tMyErr err0)
{
	tMyErr err;

	if (kReadBuffSrcXhnd != ReadBuffSrc) {
		err = kMyErrProgramBug;
	} else
	if (nullpr == ReadBuffHandle) {
		err = kMyErrProgramBug;
	} else
	{
		err = kMyErr_noErr;
	}

	ReadBuffHandle = nullpr;
	ReadBuffSrc = kReadBuffSrcNone;

#if DebugCheck
	if (kMyErr_noErr != err) {
		err = ErrReportStack0(err, "ReadBuffFromXhndEnd");
	}
#endif

	if (kMyErr_noErr != err0) {
		err = err0;
	}

	return err;
}

LOCALFUNC tMyErr ReadBuffXhndGetOffset(uimr *i)
{
	*i = ReadBuffOffSet;

	return kMyErr_noErr;
}

LOCALFUNC uimr ReadBuffXhndRemaining(uimr *v)
{
	*v = ReadBuffLength - ReadBuffOffSet;

	return kMyErr_noErr;
}

LOCALFUNC uimr ReadBuffXhndToPtr(MyPtr p, uimr L)
{
	tMyErr err;

	if (ReadBuffOffSet + L > ReadBuffLength) {
		err = kMyErrProgramBug;
	} else {
		err = kMyErr_noErr;

		MyHandleLock(ReadBuffHandle);
		MyMoveBytes(
			MyHandleOffsetToPtr(ReadBuffHandle, ReadBuffOffSet),
			p, L);
		MyHandleUnlock(ReadBuffHandle);
		ReadBuffOffSet += L;
	}

	return ErrReportStack(err, "ReadBuffXhndToPtr");
}

LOCALFUNC blnr ReadBuffXhndEOF(void)
{
	return (ReadBuffOffSet >= ReadBuffLength);
}


LOCALFUNC tMyErr ReadBuffGetOffset(uimr *i)
{
	tMyErr err;

	switch (ReadBuffSrc) {
#if UseStdFile
		case kReadBuffSrcFile:
			err = ReadBuffFileGetOffset(i);
			break;
#endif /* UseStdFile */
		case kReadBuffSrcXhnd:
			err = ReadBuffXhndGetOffset(i);
			break;
		default:
			err = kMyErrProgramBug;
			break;
	}

	return ErrReportStack(err, "ReadBuffGetOffset");
}

LOCALFUNC tMyErr ReadBuffRemaining(uimr *v)
{
	tMyErr err;

	switch (ReadBuffSrc) {
#if UseStdFile
		case kReadBuffSrcFile:
			err = ReadBuffFileRemaining(v);
			break;
#endif /* UseStdFile */
		case kReadBuffSrcXhnd:
			err = ReadBuffXhndRemaining(v);
			break;
		default:
			err = kMyErrProgramBug;
			break;
	}

	return ErrReportStack(err, "ReadBuffRemaining");
}

LOCALFUNC tMyErr ReadBuffToPtr(MyPtr p, uimr L)
{
	tMyErr err;

	switch (ReadBuffSrc) {
#if UseStdFile
		case kReadBuffSrcFile:
			err = ReadBuffFileToPtr(p, L);
			break;
#endif /* UseStdFile */
		case kReadBuffSrcXhnd:
			err = ReadBuffXhndToPtr(p, L);
			break;
		default:
			err = kMyErrProgramBug;
			break;
	}

	return ErrReportStack(err, "ReadBuffToPtr");
}

LOCALFUNC blnr ReadBuffEOF(void)
{
	blnr v;

	switch (ReadBuffSrc) {
#if UseStdFile
		case kReadBuffSrcFile:
			v = ReadBuffFileEOF();
			break;
#endif /* UseStdFile */
		case kReadBuffSrcXhnd:
			v = ReadBuffXhndEOF();
			break;
		default:
			v = trueblnr;
			break;
	}

	return v;
}


LOCALFUNC tMyErr ReadBuffToByte(ui3r *v)
{
	tMyErr err;
	ui3b buf[1];

	if (kMyErr_noErr != (err =
		ReadBuffToPtr(buf, 1)))
	{
		/* fail */
	} else
	{
		err = kMyErr_noErr;
		*v = buf[0];
	}

	return ErrReportStack(err, "ReadBuffToByte");
}

LOCALFUNC tMyErr ReadBuffUi4BE(ui4r *i)
{
	tMyErr err;
	ui3r b0;
	ui3r b1;

	if (kMyErr_noErr != (err =
		ReadBuffToByte(&b0)))
	{
		/* fail */
	} else if (kMyErr_noErr != (err =
		ReadBuffToByte(&b1)))
	{
		/* fail */
	} else {
		*i = ((ui4r)b1)
			| (((ui4r)b0) << 8);
		err = kMyErr_noErr;
	}

	return ErrReportStack(err, "ReadBuffUi4BE");
}

LOCALFUNC tMyErr ReadBuffUi5BE(ui5r *i)
{
	tMyErr err;
	ui4r b0;
	ui4r b1;

	if (kMyErr_noErr != (err =
		ReadBuffUi4BE(&b0)))
	{
		/* fail */
	} else if (kMyErr_noErr != (err =
		ReadBuffUi4BE(&b1)))
	{
		/* fail */
	} else {
		*i = ((ui5r)b1)
			| (((ui5r)b0) << 16);
		err = kMyErr_noErr;
	}

	return ErrReportStack(err, "ReadBuffUi5BE");
}

LOCALFUNC tMyErr ReadBuffUi4LE(ui4r *i)
{
	tMyErr err;
	ui3r b0;
	ui3r b1;

	if (kMyErr_noErr != (err =
		ReadBuffToByte(&b0)))
	{
		/* fail */
	} else if (kMyErr_noErr != (err =
		ReadBuffToByte(&b1)))
	{
		/* fail */
	} else {
		*i = ((ui4r)b0)
			| (((ui4r)b1) << 8);
		err = kMyErr_noErr;
	}

	return ErrReportStack(err, "ReadBuffUi4LE");
}

LOCALFUNC tMyErr ReadBuffUi5LE(ui5r *i)
{
	tMyErr err;
	ui4r b0;
	ui4r b1;

	if (kMyErr_noErr != (err =
		ReadBuffUi4LE(&b0)))
	{
		/* fail */
	} else if (kMyErr_noErr != (err =
		ReadBuffUi4LE(&b1)))
	{
		/* fail */
	} else {
		*i = ((ui5r)b0)
			| (((ui5r)b1) << 16);
		err = kMyErr_noErr;
	}

	return ErrReportStack(err, "ReadBuffUi5LE");
}

#if uimbl2sz >= 6
LOCALFUNC tMyErr ReadBuffUi6LE(ui6r *i)
{
	tMyErr err;
	ui5r b0;
	ui5r b1;

	if (kMyErr_noErr != (err =
		ReadBuffUi5LE(&b0)))
	{
		/* fail */
	} else if (kMyErr_noErr != (err =
		ReadBuffUi5LE(&b1)))
	{
		/* fail */
	} else {
		*i = ((ui6r)b0)
			| (((ui6r)b1) << 32);
		err = kMyErr_noErr;
	}

	return ErrReportStack(err, "ReadBuffUi6LE");
}
#endif

LOCALFUNC tMyErr ReadBuffSkip(uimr L)
{
	tMyErr err;
	uimr n;
	ui3b buff[1024];

label_retry:
	if (0 == L) {
		err = kMyErr_noErr;
	} else {
		n = (L > 1024) ? 1024 : L;

		if (kMyErr_noErr != (err =
			ReadBuffToPtr(buff, n)))
		{
			/* fail */
		} else {
			L -= n;

			goto label_retry;
		}
	}

	return ErrReportStack(err, "ReadBuffSkip");
}


enum {
	kWriteBuffDestNone,
#if UseStdFile
	kWriteBuffDestFile,
#endif /* UseStdFile */
	kWriteBuffDestXhnd, /* existing handle */

	CountWriteBuffDest
};

LOCALVAR ui3r WriteBuffDest = kWriteBuffDestNone;


#if UseStdFile

LOCALVAR FILE *WriteBuffFP = nullpr;

LOCALFUNC tMyErr WriteBuffToOpenFileBegin(FILE *f)
{
	tMyErr err;

	if (kWriteBuffDestNone != WriteBuffDest) {
		err = kMyErrProgramBug;
	} else
	if (nullpr != WriteBuffFP) {
		err = kMyErrProgramBug;
	} else
	{
		err = kMyErr_noErr;

		WriteBuffDest = kWriteBuffDestFile;
		WriteBuffFP = f;
	}

	return ErrReportStack(err, "WriteBuffToOpenFileBegin");
}

LOCALFUNC tMyErr WriteBuffToOpenFileEnd(void)
{
	tMyErr err;

	if (kWriteBuffDestFile != WriteBuffDest) {
		err = kMyErrProgramBug;
	} else
	if (nullpr == WriteBuffFP) {
		err = kMyErrProgramBug;
	} else
	{
		err = kMyErr_noErr;
	}

	WriteBuffFP = nullpr;
	WriteBuffDest = kWriteBuffDestNone;

	return ErrReportStack(err, "WriteBuffToOpenFileEnd");
}

LOCALFUNC tMyErr WriteBuffToFilePathBegin(char *destFile)
{
	tMyErr err;
	FILE *f;

	if ((f = fopen(destFile, "wb")) == nullpr) {
		/* Can't open files */
		err = kMyErrSysUnknown;
	} else
	if (kMyErr_noErr != (err =
		WriteBuffToOpenFileBegin(f)))
	{
		/* fail */
	} else
	{
		err = kMyErr_noErr;
	}

	return ErrReportStack(err, "WriteBuffToFilePathBegin");
}

LOCALFUNC tMyErr WriteBuffToFilePathEnd(tMyErr err0)
{
	tMyErr err;
	FILE *f = WriteBuffFP;

	if (nullpr == f) {
		err = kMyErr_noErr;
	} else {
		err = WriteBuffToOpenFileEnd();

		fflush(f);
		if (ferror(f)) {
#ifdef ENOSPC
			if (errno == ENOSPC) {
				err = kMyErr_lowdsk;
			} else
#endif
			{
				err = kMyErrSysUnknown;
			}
		}

		fclose(f);
	}

#if DebugCheck
	if (kMyErr_noErr != err) {
		err = ErrReportStack0(err, "WriteBuffToFilePathEnd");
	}
#endif

	if (kMyErr_noErr != err0) {
		err = err0;
	}

	return err;
}

LOCALFUNC tMyErr WriteBuffFileFromPtr(MyPtr p, uimr L)
{
	tMyErr err;

	if (fwrite(p, 1, L, WriteBuffFP) < L) {
		err = kMyErrSysUnknown;
	} else {
		err = kMyErr_noErr;
	}

	return ErrReportStack(err, "WriteBuffFileFromPtr");
}

#endif /* UseStdFile */


LOCALVAR MyHandle WriteBuffHandle = nullpr;
LOCALVAR uimr WriteBuffOffSet;
LOCALVAR uimr WriteBuffLength;

LOCALFUNC tMyErr WriteBuffXhndBegin(void)
{
	tMyErr err;
	MyHandle h;

	if (kWriteBuffDestNone != WriteBuffDest) {
		err = kMyErrProgramBug;
	} else
	if (nullpr != WriteBuffHandle) {
		err = kMyErrProgramBug;
	} else
	if (kMyErr_noErr != (err =
		MyHandleNew_v2(0, &h)))
	{
		/* fail */
	} else
	{
		err = kMyErr_noErr;

		WriteBuffDest = kWriteBuffDestXhnd;
		WriteBuffHandle = h;
		WriteBuffOffSet = 0;
		WriteBuffLength = 0;
	}

	return ErrReportStack(err, "WriteBuffXhndBegin");
}

LOCALFUNC tMyErr WriteBuffXhndEnd(tMyErr err0, MyHandle *h)
{
	tMyErr err;

	if (kWriteBuffDestXhnd != WriteBuffDest) {
		err = kMyErrProgramBug;
	} else
	if (nullpr == WriteBuffHandle) {
		err = kMyErrProgramBug;
	} else
	{
		if (kMyErr_noErr != err0) {
			err = err0;
		} else {
			if (WriteBuffOffSet < WriteBuffLength) {
				err = MyHandleSetSize_v2(
					WriteBuffHandle, WriteBuffOffSet);
			} else {
				err = kMyErr_noErr;
			}
		}

		if (kMyErr_noErr != err) {
			MyHandleDispose_v2(WriteBuffHandle);
		} else {
			*h = WriteBuffHandle;
		}

	}

	WriteBuffHandle = nullpr;
	WriteBuffDest = kWriteBuffDestNone;

	return ErrReportStack(err, "WriteBuffXhndEnd");
}

LOCALFUNC tMyErr WriteBuffXhndFromPtr(MyPtr p, uimr L)
{
	tMyErr err;
	uimr NewL = WriteBuffOffSet + L;

	if (NewL > WriteBuffLength) {
		uimr NewBuffLength = CeilPowerOf2(NewL);

		if (kMyErr_noErr != (err =
			MyHandleSetSize_v2(WriteBuffHandle, NewBuffLength)))
		{
			/* fail */
			goto l_exit;
		} else {
			WriteBuffLength = NewBuffLength;
		}
	}

	err = kMyErr_noErr;

	MyHandleLock(WriteBuffHandle);
	MyMoveBytes(p,
		MyHandleOffsetToPtr(WriteBuffHandle, WriteBuffOffSet), L);
	MyHandleUnlock(WriteBuffHandle);

	WriteBuffOffSet = NewL;

l_exit:
	return ErrReportStack(err, "WriteBuffXhndFromPtr");
}

LOCALFUNC tMyErr WriteBuffFromPtr(MyPtr p, uimr L)
{
	tMyErr err;

	switch (WriteBuffDest) {
#if UseStdFile
		case kWriteBuffDestFile:
			err = WriteBuffFileFromPtr(p, L);
			break;
#endif /* UseStdFile */
		case kWriteBuffDestXhnd:
			err = WriteBuffXhndFromPtr(p, L);
			break;
		default:
			err = kMyErrProgramBug;
			break;
	}

	return ErrReportStack(err, "WriteBuffFromPtr");
}

LOCALFUNC tMyErr WriteBuffFromByte(ui3r v)
{
	tMyErr err;
	ui3b buf = v;

	if (kMyErr_noErr != (err =
		WriteBuffFromPtr(&buf, 1)))
	{
		/* fail */
	} else
	{
		err = kMyErr_noErr;
	}

	return ErrReportStack(err, "WriteBuffFromByte");
}

LOCALFUNC tMyErr WriteBuffUi4BE(ui4r i)
{
	tMyErr err;
	ui3r b0 = (i >> 8) & 0xFF;
	ui3r b1 = (i     ) & 0xFF;

	if (kMyErr_noErr != (err =
		WriteBuffFromByte(b0)))
	{
		/* fail */
	} else if (kMyErr_noErr != (err =
		WriteBuffFromByte(b1)))
	{
		/* fail */
	} else {
		err = kMyErr_noErr;
	}

	return ErrReportStack(err, "WriteBuffUi4BE");
}

LOCALFUNC tMyErr WriteBuffUi5BE(ui5r i)
{
	tMyErr err;
	ui4r b0 = (i >> 16) & 0xFFFF;
	ui4r b1 = (i      ) & 0xFFFF;

	if (kMyErr_noErr != (err =
		WriteBuffUi4BE(b0)))
	{
		/* fail */
	} else if (kMyErr_noErr != (err =
		WriteBuffUi4BE(b1)))
	{
		/* fail */
	} else {
		err = kMyErr_noErr;
	}

	return ErrReportStack(err, "WriteBuffUi5BE");
}

LOCALFUNC tMyErr WriteBuffUi4LE(ui4r i)
{
	tMyErr err;
	ui3r b0 = (i     ) & 0xFF;
	ui3r b1 = (i >> 8) & 0xFF;

	if (kMyErr_noErr != (err =
		WriteBuffFromByte(b0)))
	{
		/* fail */
	} else if (kMyErr_noErr != (err =
		WriteBuffFromByte(b1)))
	{
		/* fail */
	} else {
		err = kMyErr_noErr;
	}

	return ErrReportStack(err, "WriteBuffUi4LE");
}

LOCALFUNC tMyErr WriteBuffUi5LE(ui5r i)
{
	tMyErr err;
	ui4r b0 = (i      ) & 0xFFFF;
	ui4r b1 = (i >> 16) & 0xFFFF;

	if (kMyErr_noErr != (err =
		WriteBuffUi4LE(b0)))
	{
		/* fail */
	} else if (kMyErr_noErr != (err =
		WriteBuffUi4LE(b1)))
	{
		/* fail */
	} else {
		err = kMyErr_noErr;
	}

	return ErrReportStack(err, "WriteBuffUi5LE");
}

#if uimbl2sz >= 6
LOCALFUNC tMyErr WriteBuffUi6LE(ui6r i)
{
	tMyErr err;
	ui5r b0 = (i      ) & 0xFFFFFFFFUL;
	ui5r b1 = (i >> 32) & 0xFFFFFFFFUL;

	if (kMyErr_noErr != (err =
		WriteBuffUi5LE(b0)))
	{
		/* fail */
	} else if (kMyErr_noErr != (err =
		WriteBuffUi5LE(b1)))
	{
		/* fail */
	} else {
		err = kMyErr_noErr;
	}

	return ErrReportStack(err, "WriteBuffUi6LE");
}
#endif

#if 4 == uimbl2sz
#define ReadBuffUimLE ReadBuffUi4LE
#elif 5 == uimbl2sz
#define ReadBuffUimLE ReadBuffUi5LE
#elif 6 == uimbl2sz
#define ReadBuffUimLE ReadBuffUi6LE
#else
#error "not implemented"
#endif

#if 4 == uimbl2sz
#define WriteBuffUimLE WriteBuffUi4LE
#elif 5 == uimbl2sz
#define WriteBuffUimLE WriteBuffUi5LE
#elif 6 == uimbl2sz
#define WriteBuffUimLE WriteBuffUi6LE
#else
#error "not implemented"
#endif

LOCALFUNC tMyErr WriteBuffFromCStr(char *s)
{
	tMyErr err;

	err = WriteBuffFromPtr((MyPtr)s, strlen(s));

	return ErrReportStack(err, "WriteBuffFromByte");
}

LOCALFUNC tMyErr WriteBuffCharCR(void)
{
	tMyErr err;

	err = WriteBuffFromByte('\015');

	return ErrReportStack(err, "WriteBuffCharCR");
}

LOCALFUNC tMyErr WriteBuffFromCStrAndCR(char *s)
{
	tMyErr err;

	if (kMyErr_noErr != (err =
		WriteBuffFromCStr(s)))
	{
		goto l_exit;
	}

	if (kMyErr_noErr != (err =
		WriteBuffCharCR()))
	{
		goto l_exit;
	}

	err = kMyErr_noErr;

l_exit:
	return ErrReportStack(err, "WriteBuffFromCStrAndCR");
}


LOCALFUNC tMyErr ReadBuffToNewHand(uimr L, MyHandle *h0)
{
	tMyErr err;
	MyHandle h;
	uimr n;
	ui3b buff[1024];
	uimr offset = 0;

	if (kMyErr_noErr != (err =
		MyHandleNew_v2(L, &h)))
	{
		/* fail */
	} else
	{
label_retry:
		if (0 == L) {
			err = kMyErr_noErr;
		} else {
			n = (L > 1024) ? 1024 : L;

			if (kMyErr_noErr != (err =
				ReadBuffToPtr(buff, n)))
			{
				/* fail */
			} else {

				MyHandleLock(h);
				MyMoveBytes(buff, MyHandleOffsetToPtr(h, offset), n);
				MyHandleUnlock(h);

				offset += n;
				L -= n;

				goto label_retry;
			}
		}

		err = kMyErr_noErr;
		if (kMyErr_noErr != err) {
			(void) MyHandleDispose_v2(h);
		} else {
			*h0 = h;
		}
	}

	return ErrReportStack(err, "ReadBuffToNewHand");
}

LOCALFUNC tMyErr WriteBuffFromHandRng(MyHandle h,
	uimr offsetA, uimr offsetB)
{
	tMyErr err;
	uimr L = offsetB - offsetA;

	MyHandleLock(h);
	err = WriteBuffFromPtr(MyHandleOffsetToPtr(h, offsetA), L);
	MyHandleUnlock(h);

	return ErrReportStack(err, "WriteBuffFromHandRng");
}

LOCALFUNC tMyErr WriteBuffFromHand(MyHandle h)
{
	tMyErr err;
	uimr L;

	if (kMyErr_noErr == (err =
		MyHandleGetSize_v2(h, &L)))
	{
		err = WriteBuffFromHandRng(h, 0, L);
	}

	return ErrReportStack(err, "WriteBuffFromHand");
}
